﻿using Intel.Cst.Client;
using SampleApp;
using System;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace WALWOASampleApp.CallBackFunctions
{
	class FeatureCallback : IFeatureCallback
	{
		Logger logger = new Logger();

		public void OnError(FeatureType featureType, Error state)
		{
			try
			{
				logger.Error("Received error " + state.Description + " from service");
				switch (state.ErrorType)
				{
					case ErrorType.ServiceStopped:
						{
							SampleAppProgram.walwoaForm.Invoke(new MethodInvoker(delegate
							{
								SampleAppProgram.walwoaForm.ClearFeatureCheckboxes();
								MessageBox.Show(state.Description, state.ErrorType.ToString(), MessageBoxButtons.OK, MessageBoxIcon.Error);
								SampleAppProgram.walwoaForm.EnableFeatureCheckboxes();
							}));
						}
						break;
					case ErrorType.SetOptionError:
						{
							MessageBox.Show(state.Description, "Setting Option for " + featureType + "Failed", MessageBoxButtons.OK, MessageBoxIcon.Error);
							logger.Error(state.Description);
						}
						break;
					case ErrorType.IpfUnavailable:
						{
							logger.Info("CST service lost IPF connection.");
							new Thread(() =>
							{
								MessageBox.Show("CST service lost IPF connection. CST will not take action until restored.", "IPF Communication Error", MessageBoxButtons.OK);
							}).Start();
						}
						break;
					case ErrorType.HelperServiceUnavailable:
						{
							// If your app cares about the features that require the helper service (MonitorApp and Honor User in Call)
							// you can take additional action here. User switching will also cause these events to be sent.
							SampleAppProgram.walwoaForm.SetHelperState(false);
							break;
						}
					default:
						{
							switch (featureType)
							{
								case FeatureType.LOCK:
									SampleAppProgram.walwoaForm.Invoke(new MethodInvoker(delegate
									{
										SampleAppProgram.walwoaForm.walCheckBox.CheckedChanged -= SampleAppProgram.walwoaForm.checkBoxWAL_CheckedChanged;
										if (SampleAppProgram.walwoaForm.walCheckBox.CheckState == CheckState.Checked)
										{
											SampleAppProgram.walwoaForm.walCheckBox.CheckState = CheckState.Unchecked;
											SampleAppProgram.walwoaForm.UpdateWALControlsVisibility(false);
											logger.Error(state.Description);
										}
										SampleAppProgram.walwoaForm.walCheckBox.CheckedChanged += SampleAppProgram.walwoaForm.checkBoxWAL_CheckedChanged;
										MessageBox.Show(state.Description, state.ErrorType.ToString(), MessageBoxButtons.OK, MessageBoxIcon.Error);
									}));
									break;
								case FeatureType.WAKE:
									SampleAppProgram.walwoaForm.Invoke(new MethodInvoker(delegate
									{
										SampleAppProgram.walwoaForm.woaCheckBox.CheckedChanged -= SampleAppProgram.walwoaForm.checkBoxWOA_CheckedChanged;
										if (SampleAppProgram.walwoaForm.woaCheckBox.CheckState == CheckState.Checked)
										{
											SampleAppProgram.walwoaForm.woaCheckBox.CheckState = CheckState.Unchecked;
											SampleAppProgram.walwoaForm.UpdateWOAControlsVisibility(false);
											logger.Error(state.Description);
										}
										SampleAppProgram.walwoaForm.woaCheckBox.CheckedChanged += SampleAppProgram.walwoaForm.checkBoxWOA_CheckedChanged;
										MessageBox.Show(state.Description, state.ErrorType.ToString(), MessageBoxButtons.OK, MessageBoxIcon.Error);
									}));
									break;
								case FeatureType.NOLOCKONPRESENCE:
									SampleAppProgram.walwoaForm.Invoke(new MethodInvoker(delegate
									{
										SampleAppProgram.walwoaForm.nlopCheckBox.CheckedChanged -= SampleAppProgram.walwoaForm.checkBoxNlop_CheckedChanged;
										if (SampleAppProgram.walwoaForm.nlopCheckBox.CheckState == CheckState.Checked)
										{
											SampleAppProgram.walwoaForm.nlopCheckBox.CheckState = CheckState.Unchecked;
											SampleAppProgram.walwoaForm.UpdateNLoPControlsVisibility(false);
											logger.Error(state.Description);
										}
										SampleAppProgram.walwoaForm.nlopCheckBox.CheckedChanged += SampleAppProgram.walwoaForm.checkBoxNlop_CheckedChanged;
										MessageBox.Show(state.Description, state.ErrorType.ToString(), MessageBoxButtons.OK, MessageBoxIcon.Error);
									}));
									break;
								case FeatureType.ADAPTIVEDIMMING:
									SampleAppProgram.walwoaForm.Invoke(new MethodInvoker(delegate
									{
										SampleAppProgram.walwoaForm.adCheckBox.CheckedChanged -= SampleAppProgram.walwoaForm.checkBoxAd_CheckedChanged;
										if (SampleAppProgram.walwoaForm.adCheckBox.CheckState == CheckState.Checked)
										{
											SampleAppProgram.walwoaForm.adCheckBox.CheckState = CheckState.Unchecked;
											SampleAppProgram.walwoaForm.UpdateAdControlsVisibility(false);
											logger.Error(state.Description);
										}
										SampleAppProgram.walwoaForm.adCheckBox.CheckedChanged += SampleAppProgram.walwoaForm.checkBoxAd_CheckedChanged;
										MessageBox.Show(state.Description, state.ErrorType.ToString(), MessageBoxButtons.OK, MessageBoxIcon.Error);
									}));
									break;
								case FeatureType.ONLOOKERDETECTION:
									SampleAppProgram.walwoaForm.Invoke(new MethodInvoker(delegate
									{
										SampleAppProgram.walwoaForm.onlookerDetectionCheckBox.CheckedChanged -= SampleAppProgram.walwoaForm.onlookerDetectionCheckBox_CheckedChanged;
										if (SampleAppProgram.walwoaForm.onlookerDetectionCheckBox.CheckState == CheckState.Checked)
										{
											SampleAppProgram.walwoaForm.onlookerDetectionCheckBox.CheckState = CheckState.Unchecked;
											SampleAppProgram.walwoaForm.UpdateOnlookerDetectionControlsVisibility(false);
											logger.Error(state.Description);
										}
										SampleAppProgram.walwoaForm.onlookerDetectionCheckBox.CheckedChanged += SampleAppProgram.walwoaForm.onlookerDetectionCheckBox_CheckedChanged;
										MessageBox.Show(state.Description, state.ErrorType.ToString(), MessageBoxButtons.OK, MessageBoxIcon.Error);
									}));
									break;
								case FeatureType.APPLICATION:
									SampleAppProgram.walwoaForm.Invoke(new MethodInvoker(delegate
									{
										logger.Error(state.Description);
										MessageBox.Show(state.Description, state.ErrorType.ToString(), MessageBoxButtons.OK, MessageBoxIcon.Error);
									}));
									break;
							}
						}
						break;
				}
			}
			catch (Exception e)
			{
				logger.Debug(e.Message);
				logger.Debug("Exception Thrown in WAL OnError");
			}

		}

		public void OnEvent(FeatureType featureType, EventType eventType, object eventPayload)
		{
			try
			{
				logger.Info("Event " + eventType + " for feature " + featureType + " received from service. Event payload " + eventPayload?.ToString());
				switch (eventType)
				{
					case EventType.EVENT_EXTERNAL_MONITOR:
						{
							bool externalMonitorConnected = (bool)eventPayload;
							string message = externalMonitorConnected ? "connected" : "disconnected";

							new Thread(() =>
							{
								MessageBox.Show($"External Monitor {message}", "External Monitor", MessageBoxButtons.OK);
							}).Start();
						}
						break;
					case EventType.EVENT_WAL:
						{
							//logger.Info("OnEvent: WAL in action " + eventType.ToString());
							new Thread(() =>
							{
								MessageBox.Show("WAL Event", "WAL", MessageBoxButtons.OK);
							}).Start();
						}
						break;
					case EventType.EVENT_WOA:
						{
							//logger.Info("OnEven: WOA in action " + eventType.ToString());
							new Thread(() =>
							{
								MessageBox.Show("WOA Event", "WOA", MessageBoxButtons.OK);
							}).Start();
						}
						break;
					case EventType.PROXIMITY:
						{
							float proximityDistanceMm =  (float) eventPayload;
							SampleAppProgram.walwoaForm.proximityDistanceValueLabel.Text = proximityDistanceMm.ToString();
						}
						break;
					case EventType.EVENT_SERVICE_RECOVERED:
						{
							logger.Debug("CST service recovered, refreshing features");
							SampleAppProgram.walwoaForm.Invoke(new MethodInvoker(delegate
							{
								SampleAppProgram.walwoaForm.EnableFeatureCheckboxes(true);
								_ = SampleAppProgram.walwoaForm.RefreshFeaturesViaGetOptions();
								_ = Task.Run(async () => await SampleAppProgram.walwoaForm.GetAllFeatureCapabilities());
							}));
						}
						break;
					case EventType.EVENT_ONLOOKER_DETECTED:
						{
							// This check is to prevent redundant pop-ups if an onlooker event happens before the initial state has been pulled in via capabilities
							// You usually don't need this kind of check if you aren't using pop-ups.
							if (SampleAppProgram.walwoaForm.CheckedInitialOnlookerState)
							{
								bool onlookerPresent = (bool)eventPayload;
								string message = onlookerPresent ? "detected" : "clear";

								if (onlookerPresent)
								{
									SampleAppProgram.walwoaForm.ShowOnlookerPopup();
								}
								else
								{
									SampleAppProgram.walwoaForm.HandleOnlookerClear(message);
								}
							}
						}
						break;
					case EventType.EVENT_ADAPTIVE_DIMMING:
						{
							bool adaptiveDimmingActive = (bool)eventPayload;
							string message = adaptiveDimmingActive ? "started" : "stopped";

							new Thread(() =>
							{
								MessageBox.Show($"Adaptive dimming {message}", "Adaptive Dimming", MessageBoxButtons.OK);
							}).Start();
						}
						break;
					case EventType.EVENT_CST_CAPABILITY_CHANGED:
					case EventType.EVENT_CST_EXTENDED_CAPABILITY_CHANGED:
						{
							try
							{
								new Thread(() =>
								{
								//query for status
								SampleAppProgram.walwoaForm.Invoke(new MethodInvoker(delegate
									{
										_ = Task.Run(async() => await SampleAppProgram.walwoaForm.RefreshFeatureCapability(featureType, eventType));
									}));
								}).Start();
							}
							catch (Exception ex)
							{
								logger.Debug("Unable to get feature capability for feature " + featureType + " received from service. Exception = " + ex.Message);
							}
						}
						break;
					case EventType.EVENT_IPF_STARTED:
						{
							logger.Info("CST service's IPF connection recovered.");
							new Thread(() =>
							{
								MessageBox.Show("CST service's IPF connection recovered.", "IPF Communication Restored", MessageBoxButtons.OK);
							}).Start();

							// Refresh feature capabilities
							_ = Task.Run(async () => await SampleAppProgram.walwoaForm.GetAllFeatureCapabilities());
						}
						break;
					case EventType.EVENT_SNOOZE_COMPLETE:
						{
							logger.Info("Received Snooze Complete event.");
							SampleAppProgram.walwoaForm.HandleSnoozeComplete();
						}
						break;
					case EventType.EVENT_BIOMETRIC_SENSOR_STATUS_CHANGED:
						{
							bool sensorReady = (bool)eventPayload;
							string message = sensorReady ? "ready" : "not ready";

							new Thread(() =>
							{
								MessageBox.Show($"Biometric presence sensor is now {message}", "Biometric Presence Sensor Status", MessageBoxButtons.OK);
							}).Start();
							break;
						}
					case EventType.EVENT_HELPER_SERVICE_AVAILABLE:
						{
							SampleAppProgram.walwoaForm.SetHelperState(true);
							break;
						}
					case EventType.EVENT_HUMAN_PRESENCE_CHANGED:
						{
							if(!SampleAppProgram.walwoaForm.HumanPresencePollEnabled)
							{
								HumanPresenceStatusPacket updatedPresence = (HumanPresenceStatusPacket)eventPayload;
								SampleAppProgram.walwoaForm.UpdatePresenceStatusText(updatedPresence.PlatformStatus.ToString(), updatedPresence.PolicyStatus.ToString(), updatedPresence.SensorStatus.ToString());
							}
							break;
						}
				}
			}
			catch (Exception e)
			{
				logger.Debug(e.Message);
				logger.Error("Exception on handling OnEvent");
			}
		}

		public void OnSuccess(FeatureType featureType, ResponseType responseType)
		{
			try
			{
				logger.Info("OnSuccess with response " + responseType.ToString() + " from service, for feature "+featureType.ToString());
				switch (featureType)
				{
					case FeatureType.LOCK:
						SampleAppProgram.walwoaForm.walCheckBox.Enabled = true;
						break;
					case FeatureType.WAKE:
						SampleAppProgram.walwoaForm.woaCheckBox.Enabled = true;
						break;
					case FeatureType.NOLOCKONPRESENCE:
						SampleAppProgram.walwoaForm.nlopCheckBox.Enabled = true;
						break;
					case FeatureType.ADAPTIVEDIMMING:
						SampleAppProgram.walwoaForm.adCheckBox.Enabled = true;
						break;
					case FeatureType.ONLOOKERDETECTION:
						SampleAppProgram.walwoaForm.onlookerDetectionCheckBox.Enabled = true;
						break;
					case FeatureType.APPLICATION:
						logger.Info("Enable/Disable success for Privacy " + responseType.ToString());
						String successMsg = "";
						switch(responseType)
						{
							case ResponseType.NO_CHANGE:
								successMsg = "No face detection sensor found on the system. Sensor switch did not occur.";
								break;
							case ResponseType.ENABLE_SUCCESS:
								successMsg = "Enabled privacy";
								break;
							case ResponseType.DISABLE_SUCCESS:
								successMsg = "Disabled privacy";
								break;
						}
						new Thread(() =>
						{
							MessageBox.Show(successMsg, "Privacy Switch", MessageBoxButtons.OK);
						}).Start();
						break;
				}
			}
			catch (Exception e)
			{
				logger.Debug(e.Message);
				logger.Error("Exception on handling OnSuccess");
			}
		}
	}
}
