using Intel.Cst.Client;
using Microsoft.Win32;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Forms;
using WALWOASampleApp.CallBackFunctions;
using WALWOASampleApp.Properties;

namespace SampleApp
{
	public partial class FormWALWOA : Form
	{
		private static Logger logger = new Logger();

		private ClientBackendLink _commClient = null;
		private readonly int[] preDimIntervalSeconds = new int[] { 3, 5, 10, 15, 20, 25, 30, 35, 40, 45, 50, 55, 60 };
		private readonly int[] userPresentWaitIntervalsSeconds = new int[] { 5, 10 };
		private readonly int[] lockAfterDimIntervalsSeconds = new int[] { 3, 5, 10, 15, 20, 25, 30, 35, 40, 45, 50, 55, 60 };
		private readonly int[] mispredictionDimWaitSeconds = new int[] { 15, 20, 25, 30, 35, 40, 45, 50, 55, 60, 75, 90, 105, 120, 150, 180, 210, 240, 270, 300 };
		private readonly int[] mispredictionTimeWindowSeconds = new int[] { 3, 5, 7, 10 };
		private readonly int[] batteryRemainingPercentage = new int[] { 5, 10, 15, 20, 25, 30, 35, 40, 45, 50 };
		private readonly int[] dimTarget = new int[] { 0, 5, 10, 15, 20, 25 };
		private readonly int[] dimmingIntervalSeconds = new int[] { 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20 };
		private readonly int[] failSafeIntervalSeconds = new int[] { 1800, 3600 };
		private readonly int[] detectionAbsenceIntervalSeconds = new int[] { 1, 2, 3, 5, 10, 15, 20, 25, 30, 35, 40, 45, 50, 55, 60 };
		private bool serviceReconnectRequired = false;
		private bool connectedRemotely = false;
		private IFeatureCallback featureCallback;
		private SettingsAccessType globalMode = SettingsAccessType.LOCAL;
		private readonly string[] appNames = new string[] { "Intel.ContextSensingSampleApp", "Intel User Awareness Sample App", "Dell.DellOptimizer", "DellExpressSignIn" };
		private readonly string[] eventNames = EventTypeMap.EventSubscriptionNames.Values.ToArray();
		private CancellationTokenSource humanPresenceTaskTokenSource;
		private Task humanPresenceBackgroundTask = Task.CompletedTask;
		private const int HUMAN_PRESENCE_UPDATE_INTERVAL = 1500;
		const string DATA_CONNECTION_ERROR_TEXT = "FAULT"; // when displayed data depends on an active client connection (poll) but there is no connection
		const string DATA_UNAVAILABLE_TEXT = "UNAVAILABLE"; // when displayed data depends on an event but the event is not subscribed
		private bool snoozeRequested = false;
		private bool visionCapable = true;

		public bool CheckedInitialOnlookerState { get; private set; } = false;
		public bool HumanPresencePollEnabled => humanPresencePollCheckbox.Checked;

		#region constructor

		public FormWALWOA(bool bHidePrimaryGUI = false)
		{
			InitializeComponent();
			icstSampleAppVersionLabel.Text = "Intel Context Sensing SampleApp Version " + Application.ProductVersion.ToString();
			featureCallback = new FeatureCallback();
			SystemEvents.SessionSwitch += new SessionSwitchEventHandler(SystemEvents_SessionSwitch);
			if (bHidePrimaryGUI)
			{
				WindowState = FormWindowState.Minimized;
			}

			humanPresenceTaskTokenSource = new CancellationTokenSource();
			humanPresenceTaskTokenSource.Cancel();
		}

		#endregion constructor

		#region system event handlers

		private void SystemEvents_SessionSwitch(object sender, SessionSwitchEventArgs e)
		{
			logger.Debug("SessionSwitch event occured with reason = " + e.Reason);
			if (serviceReconnectRequired && !connectedRemotely &&
				(e.Reason == SessionSwitchReason.SessionLogon ||
				e.Reason == SessionSwitchReason.SessionUnlock))
			{
				serviceReconnectRequired = false;
				_ = RefreshFeaturesViaGetOptions();
				_ = GetAllFeatureCapabilities();

				if (this.HumanPresencePollEnabled)
				{
					StartHumanPresenceUpdateTask();
				}
				else
				{
					// Since our HP is out of date, call the SDK once to make sure we have the latest data.
					// After this, we'll get events if it changes as long as the app is subscribed.
					UpdatePresenceStatus();
				}
			}
			else if (!serviceReconnectRequired &&
				(e.Reason == SessionSwitchReason.SessionLogoff ||
				e.Reason == SessionSwitchReason.SessionLock))
			{
				logger.Debug("SessionSwitch: Service reconnect is required");
				serviceReconnectRequired = true;
				StopHumanPresenceUpdateTask();
			}
			else if (e.Reason == SessionSwitchReason.RemoteConnect)
			{
				connectedRemotely = true;
				walCheckBox.Enabled = false;
				woaCheckBox.Enabled = false;
				nlopCheckBox.Enabled = false;
				adCheckBox.Enabled = false;
				onlookerDetectionCheckBox.Enabled = false;
				privacyCheckbox.Enabled = false;
				monitorAppCheckbox.Enabled = false;
				StopHumanPresenceUpdateTask();
			}
			else if (e.Reason == SessionSwitchReason.RemoteDisconnect)
			{
				connectedRemotely = false;
				walCheckBox.Enabled = true;
				woaCheckBox.Enabled = true;
				nlopCheckBox.Enabled = true;
				adCheckBox.Enabled = true;
				onlookerDetectionCheckBox.Enabled = true;
				privacyCheckbox.Enabled = this.visionCapable || this.testModeCheckbox.Checked;
				monitorAppCheckbox.Enabled = true;
				// Intentionally not starting the human presence update task here
				// Defer starting it until the sessionlogon/unlock to avoid spamming ICST with requests it will reject
			}
		}

		#endregion system event handlers

		#region form event handlers

		private async void FormWALWOA_Load(object sender, EventArgs e)
		{
			CenterToScreen();
			InitializeControls();
			_commClient = new ClientBackendLink(featureCallback);

			if (this.HumanPresencePollEnabled)
			{
				StartHumanPresenceUpdateTask();
			}
			else
			{
				UpdatePresenceStatus();
			}

			bool success = await RefreshFeaturesViaGetOptions();
			
			if (success)
			{
				SetAppName();
				SubscribeToCapabilityEvents();
			}

			_ = Task.Run(async () => await GetAllFeatureCapabilities());
		}

		private void FormWALWOA_FormClosing(object sender, FormClosingEventArgs e)
		{
			StopHumanPresenceUpdateTask();
			SystemEvents.SessionSwitch -= SystemEvents_SessionSwitch;
			Settings.Default.Save();
		}

		#endregion form event handlers

		#region control event handler

		internal async void checkBoxWAL_CheckedChanged(object sender, EventArgs e)
		{
			Cursor.Current = Cursors.WaitCursor;
			walCheckBox.Enabled = false;

			if (walCheckBox.Checked)
			{
				try
				{

					if (globalMode == SettingsAccessType.GLOBAL)
					{
						logger.Info("Showing default settings for Walk Away Lock");
						await _commClient.SetOption(FeatureType.LOCK, FeatureProperty.FeatureEnabled, true, globalMode);
					}
					else
					{
						logger.Info("Walk Away Lock feature is being enabled");
						_commClient.Enable(FeatureType.LOCK);
					}
					FeatureSetting featureSetting = await _commClient.GetOptions(FeatureType.LOCK, globalMode);
					UpdateTimerControlSettings(featureSetting);
				}
				catch (Exception ex)
				{
					walCheckBox.CheckedChanged -= checkBoxWAL_CheckedChanged;
					walCheckBox.CheckState = CheckState.Unchecked;
					walCheckBox.CheckedChanged += checkBoxWAL_CheckedChanged;
					logger.Error("Error enabling Walk Away Lock " + ex.Message);
					MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
					walCheckBox.Enabled = true;
				}
			}
			else
			{
				try
				{
					if (globalMode == SettingsAccessType.GLOBAL)
					{
						logger.Info("Setting global disable for Walk Away Lock");
						await _commClient.SetOption(FeatureType.LOCK, FeatureProperty.FeatureEnabled, false, globalMode);
						FeatureSetting featureSetting = await _commClient.GetOptions(FeatureType.LOCK, globalMode);
						UpdateTimerControlSettings(featureSetting);
					}
					else
					{
						logger.Info("Walk Away Lock feature is being disabled");
						_commClient.Disable(FeatureType.LOCK);
					}
				}
				catch (Exception ex)
				{
					walCheckBox.CheckedChanged -= checkBoxWAL_CheckedChanged;
					walCheckBox.CheckState = CheckState.Checked;
					walCheckBox.CheckedChanged += checkBoxWAL_CheckedChanged;
					logger.Error("Error disabling Walk Away Lock " + ex.Message);
					MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
					walCheckBox.Enabled = true;
				}
			}

			Cursor.Current = Cursors.Default;
		}

		internal async void checkBoxWOA_CheckedChanged(object sender, EventArgs e)
		{
			Cursor.Current = Cursors.WaitCursor;
			woaCheckBox.Enabled = false;

			if (woaCheckBox.Checked)
			{
				try
				{
					if (globalMode == SettingsAccessType.GLOBAL)
					{
						logger.Info("Showing default settings for Wake On Approach");
						await _commClient.SetOption(FeatureType.WAKE, FeatureProperty.FeatureEnabled, true, globalMode);
					}
					else
					{
						logger.Info("Wake On Approach feature is being enabled");
						_commClient.Enable(FeatureType.WAKE);
					}

					FeatureSetting featureSetting = await _commClient.GetOptions(FeatureType.WAKE, globalMode);
					UpdateTimerControlSettings(featureSetting);
				}
				catch (Exception ex)
				{
					woaCheckBox.CheckedChanged -= checkBoxWOA_CheckedChanged;
					woaCheckBox.CheckState = CheckState.Unchecked;
					woaCheckBox.CheckedChanged += checkBoxWOA_CheckedChanged;
					logger.Error("Error enabling Wake On Approach " + ex.Message);
					MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
					woaCheckBox.Enabled = true;
				}
			}
			else
			{
				try
				{
					if (globalMode == SettingsAccessType.GLOBAL)
					{
						logger.Info("Setting global disable for Wake On Approach");
						await _commClient.SetOption(FeatureType.WAKE, FeatureProperty.FeatureEnabled, false, globalMode);
						FeatureSetting featureSetting = await _commClient.GetOptions(FeatureType.WAKE, globalMode);
						UpdateTimerControlSettings(featureSetting);
					}
					else
					{
						logger.Info("Wake On Approach feature is being disabled");
						_commClient.Disable(FeatureType.WAKE);
					}

				}
				catch (Exception ex)
				{
					woaCheckBox.CheckedChanged -= checkBoxWOA_CheckedChanged;
					woaCheckBox.CheckState = CheckState.Checked;
					woaCheckBox.CheckedChanged += checkBoxWOA_CheckedChanged;
					logger.Error("Error disabling Wake On Approach " + ex.Message);
					MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
					woaCheckBox.Enabled = true;
				}
			}

			Cursor.Current = Cursors.Default;
		}

		internal async void checkBoxNlop_CheckedChanged(object sender, EventArgs e)
		{
			Cursor.Current = Cursors.WaitCursor;
			nlopCheckBox.Enabled = false;

			if (nlopCheckBox.Checked)
			{
				try
				{
					if (globalMode == SettingsAccessType.GLOBAL)
					{
						logger.Info("Showing default settings for No Lock On Presence");
						await _commClient.SetOption(FeatureType.NOLOCKONPRESENCE, FeatureProperty.FeatureEnabled, true, globalMode);
					}
					else
					{
						logger.Info("No lock on presence feature is being enabled");
						_commClient.Enable(FeatureType.NOLOCKONPRESENCE);
					}
					FeatureSetting featureSetting = await _commClient.GetOptions(FeatureType.NOLOCKONPRESENCE, globalMode);
					UpdateTimerControlSettings(featureSetting);
				}
				catch (Exception ex)
				{
					nlopCheckBox.CheckedChanged -= checkBoxNlop_CheckedChanged;
					nlopCheckBox.CheckState = CheckState.Unchecked;
					nlopCheckBox.CheckedChanged += checkBoxNlop_CheckedChanged;
					logger.Error("Error enabling No Lock on Presence " + ex.Message);
					MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
					nlopCheckBox.Enabled = true;
				}
			}
			else
			{
				try
				{
					if (globalMode == SettingsAccessType.GLOBAL)
					{
						logger.Info("Showing default settings for No Lock On Presence");
						await _commClient.SetOption(FeatureType.NOLOCKONPRESENCE, FeatureProperty.FeatureEnabled, false, globalMode);
						FeatureSetting featureSetting = await _commClient.GetOptions(FeatureType.NOLOCKONPRESENCE, globalMode);
						UpdateTimerControlSettings(featureSetting);
					}
					else
					{
						logger.Info("No lock on presence feature is being disabled");
						_commClient.Disable(FeatureType.NOLOCKONPRESENCE);
					}
				}
				catch (Exception ex)
				{
					nlopCheckBox.CheckedChanged -= checkBoxNlop_CheckedChanged;
					nlopCheckBox.CheckState = CheckState.Checked;
					nlopCheckBox.CheckedChanged += checkBoxNlop_CheckedChanged;
					logger.Error("Error disabling No Lock on Presence " + ex.Message);
					MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
					nlopCheckBox.Enabled = true;
				}
			}

			Cursor.Current = Cursors.Default;
		}

		internal async void checkBoxAd_CheckedChanged(object sender, EventArgs e)
		{
			Cursor.Current = Cursors.WaitCursor;
			adCheckBox.Enabled = false;

			if (adCheckBox.Checked)
			{
				try
				{
					if (globalMode == SettingsAccessType.GLOBAL)
					{
						logger.Info("Showing default settings for Adaptive Dimming");
						await _commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.FeatureEnabled, true, globalMode);
					}
					else
					{
						logger.Info("Adaptive dimming feature is being enabled");
						_commClient.Enable(FeatureType.ADAPTIVEDIMMING);
					}
					FeatureSetting featureSetting = await _commClient.GetOptions(FeatureType.ADAPTIVEDIMMING, globalMode);
					UpdateTimerControlSettings(featureSetting);
				}
				catch (Exception ex)
				{
					adCheckBox.CheckedChanged -= checkBoxAd_CheckedChanged;
					adCheckBox.CheckState = CheckState.Unchecked;
					adCheckBox.CheckedChanged += checkBoxAd_CheckedChanged;
					logger.Error("Error enabling Adaptive Dimming " + ex.Message);
					MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
					adCheckBox.Enabled = true;
				}
			}
			else
			{
				try
				{
					if (globalMode == SettingsAccessType.GLOBAL)
					{
						logger.Info("Showing default settings for Adaptive Dimming");
						await _commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.FeatureEnabled, false, globalMode);
						FeatureSetting featureSetting = await _commClient.GetOptions(FeatureType.ADAPTIVEDIMMING, globalMode);
						UpdateTimerControlSettings(featureSetting);
					}
					else
					{
						logger.Info("Adaptive dimming feature is being disabled");
						_commClient.Disable(FeatureType.ADAPTIVEDIMMING);
					}

				}
				catch (Exception ex)
				{
					adCheckBox.CheckedChanged -= checkBoxAd_CheckedChanged;
					adCheckBox.CheckState = CheckState.Checked;
					adCheckBox.CheckedChanged += checkBoxAd_CheckedChanged;
					logger.Error("Error disabling Adaptive Dimming " + ex.Message);
					MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
					adCheckBox.Enabled = true;
				}
			}

			Cursor.Current = Cursors.Default;
		}

		internal async void onlookerDetectionCheckBox_CheckedChanged(object sender, EventArgs e)
		{
			Cursor.Current = Cursors.WaitCursor;
			onlookerDetectionCheckBox.Enabled = false;

			if (onlookerDetectionCheckBox.Checked)
			{
				try
				{

					if (globalMode == SettingsAccessType.GLOBAL)
					{
						logger.Info("Showing default settings for Onlooker Detection");
						await _commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.FeatureEnabled, true, globalMode);
					}
					else
					{
						logger.Info("Onlooker Detection feature is being enabled");
						_commClient.Enable(FeatureType.ONLOOKERDETECTION);
					}
					FeatureSetting featureSetting = await _commClient.GetOptions(FeatureType.ONLOOKERDETECTION, globalMode);
					UpdateTimerControlSettings(featureSetting);
				}
				catch (Exception ex)
				{
					onlookerDetectionCheckBox.CheckedChanged -= onlookerDetectionCheckBox_CheckedChanged;
					onlookerDetectionCheckBox.CheckState = CheckState.Unchecked;
					onlookerDetectionCheckBox.CheckedChanged += onlookerDetectionCheckBox_CheckedChanged;
					logger.Error("Error enabling onlooker detection " + ex.Message);
					MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
					onlookerDetectionCheckBox.Enabled = true;
				}
			}
			else
			{
				try
				{
					if (globalMode == SettingsAccessType.GLOBAL)
					{
						logger.Info("Setting global disable for Onlooker Detection");
						await _commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.FeatureEnabled, false, globalMode);
						FeatureSetting featureSetting = await _commClient.GetOptions(FeatureType.ONLOOKERDETECTION, globalMode);
						UpdateTimerControlSettings(featureSetting);
					}
					else
					{
						logger.Info("Onlooker Detection feature is being disabled");
						_commClient.Disable(FeatureType.ONLOOKERDETECTION);
					}
				}
				catch (Exception ex)
				{
					onlookerDetectionCheckBox.CheckedChanged -= onlookerDetectionCheckBox_CheckedChanged;
					onlookerDetectionCheckBox.CheckState = CheckState.Checked;
					onlookerDetectionCheckBox.CheckedChanged += onlookerDetectionCheckBox_CheckedChanged;
					logger.Error("Error disabling Onlooker Detection " + ex.Message);
					MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
					onlookerDetectionCheckBox.Enabled = true;
				}
			}

			Cursor.Current = Cursors.Default;
		}

		private void checkBoxWAL_EnabledChanged(object sender, EventArgs e)
		{
			if (globalMode == SettingsAccessType.LOCAL)
			{
				if (walCheckBox.Enabled && walCheckBox.Checked)
					UpdateWALControlsVisibility(true);
				else
					UpdateWALControlsVisibility(false);
			}
		}

		private void checkBoxWOA_EnabledChanged(object sender, EventArgs e)
		{
			if (globalMode == SettingsAccessType.LOCAL)
			{
				if (woaCheckBox.Enabled && woaCheckBox.Checked)
					UpdateWOAControlsVisibility(true);
				else
					UpdateWOAControlsVisibility(false);
			}
		}

		private void checkBoxNlop_EnabledChanged(object sender, EventArgs e)
		{
			if (globalMode == SettingsAccessType.LOCAL)
			{
				if (nlopCheckBox.Enabled && nlopCheckBox.Checked)
					UpdateNLoPControlsVisibility(true);
				else
					UpdateNLoPControlsVisibility(false);
			}
		}

		private void checkBoxAd_EnabledChanged(object sender, EventArgs e)
		{
			if (globalMode == SettingsAccessType.LOCAL)
			{
				if (adCheckBox.Enabled && adCheckBox.Checked)
					UpdateAdControlsVisibility(true);
				else
					UpdateAdControlsVisibility(false);
			}
		}

		private void onlookerDetectionCheckBox_EnabledChanged(object sender, EventArgs e)
		{
			if (globalMode == SettingsAccessType.LOCAL)
			{
				if (onlookerDetectionCheckBox.Enabled && onlookerDetectionCheckBox.Checked)
					UpdateOnlookerDetectionControlsVisibility(true);
				else
					UpdateOnlookerDetectionControlsVisibility(false);
			}
		}

		private void UpdateTimerControlSettings(FeatureSetting featureSetting)
		{
			if (featureSetting is LockFeatures)
			{
				preDimIntervalComboBox.SelectedIndexChanged -= new System.EventHandler(this.preDimIntervalComboBox_SelectedIndexChanged);
				preDimIntervalComboBox.SelectedIndex = Array.IndexOf(preDimIntervalSeconds, ((LockFeatures)featureSetting).PreDimInterval / 1000);
				preDimIntervalComboBox.SelectedIndexChanged += new System.EventHandler(this.preDimIntervalComboBox_SelectedIndexChanged);

				notPresentDimmingTimeComboBox.SelectedIndexChanged -= new System.EventHandler(this.notPresentDimmingTimeComboBox_SelectedIndexChanged);
				notPresentDimmingTimeComboBox.SelectedIndex = Array.IndexOf(dimmingIntervalSeconds, ((LockFeatures)featureSetting).DimmingInterval / 1000);
				notPresentDimmingTimeComboBox.SelectedIndexChanged += new System.EventHandler(this.notPresentDimmingTimeComboBox_SelectedIndexChanged);

				userPresentWaitIntervalComboBox.SelectedIndexChanged -= new System.EventHandler(this.userPresentWaitIntervalComboBox_SelectedIndexChanged);
				userPresentWaitIntervalComboBox.SelectedIndex = Array.IndexOf(userPresentWaitIntervalsSeconds, ((LockFeatures)featureSetting).WaitInterval / 1000);
				userPresentWaitIntervalComboBox.SelectedIndexChanged += new System.EventHandler(this.userPresentWaitIntervalComboBox_SelectedIndexChanged);

				lockAfterDimIntervalComboBox.SelectedIndexChanged -= new System.EventHandler(this.lockIntervalComboBox_SelectedIndexChanged);
				lockAfterDimIntervalComboBox.SelectedIndex = Array.IndexOf(lockAfterDimIntervalsSeconds, ((LockFeatures)featureSetting).LockAfterDimInterval / 1000);
				lockAfterDimIntervalComboBox.SelectedIndexChanged += new System.EventHandler(this.lockIntervalComboBox_SelectedIndexChanged);
			}
			else if (featureSetting is NoLockOnPresenceFeatures)
			{
				comboBoxWaitTimer.SelectedIndexChanged -= new System.EventHandler(this.waitIntervalComboBox_SelectedIndexChanged);
				comboBoxWaitTimer.SelectedIndex = Array.IndexOf(preDimIntervalSeconds, ((NoLockOnPresenceFeatures)featureSetting).WaitInterval / 1000);
				comboBoxWaitTimer.SelectedIndexChanged += new System.EventHandler(this.waitIntervalComboBox_SelectedIndexChanged);

				nlopFailSafeTimeoutComboBox.SelectedIndexChanged -= new System.EventHandler(this.failSafeTimeoutComboBox_SelectedIndexChanged);
				nlopFailSafeTimeoutComboBox.SelectedIndex = Array.IndexOf(failSafeIntervalSeconds, ((NoLockOnPresenceFeatures)featureSetting).FailSafeTimeout / 1000);
				nlopFailSafeTimeoutComboBox.SelectedIndexChanged += new System.EventHandler(this.failSafeTimeoutComboBox_SelectedIndexChanged);
			}
			else if (featureSetting is AdaptiveDimmingFeatures)
			{
				adNotPresentDimWaitTimeComboBox.SelectedIndexChanged -= new System.EventHandler(this.adNotPresentDimWaitTimeComboBox_SelectedIndexChanged);
				adNotPresentDimWaitTimeComboBox.SelectedIndex = Array.IndexOf(preDimIntervalSeconds, ((AdaptiveDimmingFeatures)featureSetting).NotPresentDimWaitTime / 1000);
				adNotPresentDimWaitTimeComboBox.SelectedIndexChanged += new System.EventHandler(this.adNotPresentDimWaitTimeComboBox_SelectedIndexChanged);

				mispredictionTimeWindowCombobox.SelectedIndexChanged -= new System.EventHandler(this.mispredictionTimeWindowCombobox_SelectedIndexChanged);
				mispredictionTimeWindowCombobox.SelectedIndex = Array.IndexOf(mispredictionTimeWindowSeconds, ((AdaptiveDimmingFeatures)featureSetting).MispredictionTimeWindow / 1000);
				mispredictionTimeWindowCombobox.SelectedIndexChanged += new System.EventHandler(this.mispredictionTimeWindowCombobox_SelectedIndexChanged);

				MP1DimWaitComboBox.SelectedIndexChanged -= new System.EventHandler(this.MP1DimWaitComboBox_SelectedIndexChanged);
				MP1DimWaitComboBox.SelectedIndex = Array.IndexOf(mispredictionDimWaitSeconds, ((AdaptiveDimmingFeatures)featureSetting).Misprediction1DimWaitTime / 1000);
				MP1DimWaitComboBox.SelectedIndexChanged += new System.EventHandler(this.MP1DimWaitComboBox_SelectedIndexChanged);

				MP2DimWaitComboBox.SelectedIndexChanged -= new System.EventHandler(this.MP2DimWaitComboBox_SelectedIndexChanged);
				MP2DimWaitComboBox.SelectedIndex = Array.IndexOf(mispredictionDimWaitSeconds, ((AdaptiveDimmingFeatures)featureSetting).Misprediction2DimWaitTime / 1000);
				MP2DimWaitComboBox.SelectedIndexChanged += new System.EventHandler(this.MP2DimWaitComboBox_SelectedIndexChanged);

				MP3DimWaitComboBox.SelectedIndexChanged -= new System.EventHandler(this.MP3DimWaitComboBox_SelectedIndexChanged);
				MP3DimWaitComboBox.SelectedIndex = Array.IndexOf(mispredictionDimWaitSeconds, ((AdaptiveDimmingFeatures)featureSetting).Misprediction3DimWaitTime / 1000);
				MP3DimWaitComboBox.SelectedIndexChanged += new System.EventHandler(this.MP3DimWaitComboBox_SelectedIndexChanged);

				MP4DimWaitComboBox.SelectedIndexChanged -= new System.EventHandler(this.MP4DimWaitComboBox_SelectedIndexChanged);
				MP4DimWaitComboBox.SelectedIndex = Array.IndexOf(mispredictionDimWaitSeconds, ((AdaptiveDimmingFeatures)featureSetting).Misprediction4DimWaitTime / 1000);
				MP4DimWaitComboBox.SelectedIndexChanged += new System.EventHandler(this.MP4DimWaitComboBox_SelectedIndexChanged);

				adDimmingIntervalComboBox.SelectedIndexChanged -= new System.EventHandler(this.adDimmingIntervalComboBox_SelectedIndexChanged);
				adDimmingIntervalComboBox.SelectedIndex = Array.IndexOf(dimmingIntervalSeconds, ((AdaptiveDimmingFeatures)featureSetting).DimmingInterval / 1000);
				adDimmingIntervalComboBox.SelectedIndexChanged += new System.EventHandler(this.adDimmingIntervalComboBox_SelectedIndexChanged);

				adDisengagedDimWaitTimeComboBox.SelectedIndexChanged -= new System.EventHandler(this.adDisengagedDimWaitTimeComboBox_SelectedIndexChanged);
				adDisengagedDimWaitTimeComboBox.SelectedIndex = Array.IndexOf(preDimIntervalSeconds, ((AdaptiveDimmingFeatures)featureSetting).DisengagedDimWaitTime / 1000);
				adDisengagedDimWaitTimeComboBox.SelectedIndexChanged += new System.EventHandler(this.adDisengagedDimWaitTimeComboBox_SelectedIndexChanged);
			}
			else if (featureSetting is OnlookerDetectionFeatures)
			{
				onlookerDetectionDelayIntervalComboBox.SelectedIndexChanged -= new System.EventHandler(this.onlookerDetectionDelayIntervalComboBox_SelectedIndexChanged);
				onlookerDetectionDelayIntervalComboBox.SelectedIndex = Array.IndexOf(preDimIntervalSeconds, ((OnlookerDetectionFeatures)featureSetting).DetectionDelayInterval / 1000);
				onlookerDetectionDelayIntervalComboBox.SelectedIndexChanged += new System.EventHandler(this.onlookerDetectionDelayIntervalComboBox_SelectedIndexChanged);

				onlookerDetectionDimmingTimeComboBox.SelectedIndexChanged -= new System.EventHandler(this.onlookerDetectionDimmingTimeComboBox_SelectedIndexChanged);
				onlookerDetectionDimmingTimeComboBox.SelectedIndex = Array.IndexOf(dimmingIntervalSeconds, ((OnlookerDetectionFeatures)featureSetting).DimmingInterval / 1000);
				onlookerDetectionDimmingTimeComboBox.SelectedIndexChanged += new System.EventHandler(this.onlookerDetectionDimmingTimeComboBox_SelectedIndexChanged);

				onlookerDetectionAbsenceIntervalComboBox.SelectedIndexChanged -= new System.EventHandler(this.onlookerDetectionAbsenceIntervalComboBox_SelectedIndexChanged);
				onlookerDetectionAbsenceIntervalComboBox.SelectedIndex = Array.IndexOf(detectionAbsenceIntervalSeconds, ((OnlookerDetectionFeatures)featureSetting).OnlookerAbsenceWaitInterval / 1000);
				onlookerDetectionAbsenceIntervalComboBox.SelectedIndexChanged += new System.EventHandler(this.onlookerDetectionAbsenceIntervalComboBox_SelectedIndexChanged);
			}
		}

		#endregion control event handler

		#region GUI initialization

		private void InitializeControls()
		{
			LoadControlPresets();
			AddToolTipsToControls();
			UpdateAppFeatureControlsVisibility(testModeCheckbox.Checked); // Make sure that the default visibility matches the default state of the test mode checkbox
		}

		private void AddToolTipsToControls()
		{
			string awayString = "Have PC lock and turn off display when you move away from your PC";
			string awayExternalMontitorString = "Keep WAL enabled while connected to external monitor(s)";
			string preDimIntervalString = "Time to dim after you leave";
			string lockAfterDimString = "Time to lock after dim";
			string nearString = "Have PC wake as you approach";
			string nearExternalMonitorString = "Keep WOA enabled while connected to external monitor(s)";
			string proximityDistanceString = "Real Time Proximity Sensor reading (in mm)";
			using (var toolTip = new ToolTip())
			{
				toolTip.AutoPopDelay = 5000;
				toolTip.ReshowDelay = 500;
				toolTip.ShowAlways = true;
				toolTip.SetToolTip(this.walLabel, awayString);
				toolTip.SetToolTip(this.walCheckBox, awayString);
				toolTip.SetToolTip(this.walPanel, awayString);
				toolTip.SetToolTip(this.walExternalMonitorLabel, awayExternalMontitorString);
				toolTip.SetToolTip(this.walExternalMonitorOnRadioButton, awayExternalMontitorString);
				toolTip.SetToolTip(this.walExternalMonitorOffRadioButton, awayExternalMontitorString);
				toolTip.SetToolTip(this.preDimIntervalLabel, preDimIntervalString);
				toolTip.SetToolTip(this.preDimIntervalComboBox, preDimIntervalString);
				toolTip.SetToolTip(this.lockAfterDimIntervalLabel, lockAfterDimString);
				toolTip.SetToolTip(this.lockAfterDimIntervalComboBox, lockAfterDimString);
				toolTip.SetToolTip(this.woaLabel, nearString);
				toolTip.SetToolTip(this.woaCheckBox, nearString);
				toolTip.SetToolTip(this.woaPanel, nearString);
				toolTip.SetToolTip(this.panelWoa, nearExternalMonitorString);
				toolTip.SetToolTip(this.woaExternalMonitorLabel, nearExternalMonitorString);
				toolTip.SetToolTip(this.woaExternalMonitorOnRadioButton, nearExternalMonitorString);
				toolTip.SetToolTip(this.woaExternalMonitorOffRadioButton, nearExternalMonitorString);
				toolTip.SetToolTip(this.proximityDistanceLabel, proximityDistanceString);
				toolTip.SetToolTip(this.proximityDistancePanel, proximityDistanceString);
				toolTip.SetToolTip(this.proximityDistanceValueLabel, proximityDistanceString);
			}
		}

		private void AddControlEventHandlers()
		{
			/* Do not remove event handler assignments! At this point, the event handlers for the controls have not been added. */

			walCheckBox.CheckedChanged += new System.EventHandler(this.checkBoxWAL_CheckedChanged);
			walExternalMonitorOnRadioButton.CheckedChanged += new System.EventHandler(this.awayExternalMonitorOnRadioButton_CheckedChanged);
			walExternalMonitorOffRadioButton.CheckedChanged += new System.EventHandler(this.awayExternalMonitorOffRadioButton_CheckedChanged);
			preDimIntervalComboBox.SelectedIndexChanged += new System.EventHandler(this.preDimIntervalComboBox_SelectedIndexChanged);
			notPresentDimmingTimeComboBox.SelectedIndexChanged += new System.EventHandler(this.notPresentDimmingTimeComboBox_SelectedIndexChanged);
			notPresentDimTargetComboBox.SelectedIndexChanged += new System.EventHandler(this.notPresentDimTargetComboBox_SelectedIndexChanged);
			userPresentWaitIntervalComboBox.SelectedIndexChanged += new System.EventHandler(this.userPresentWaitIntervalComboBox_SelectedIndexChanged);
			dimScreenOnRadioButton.CheckedChanged += new System.EventHandler(this.dimScreenOnRadioButton_CheckedChanged);
			dimScreenOffRadioButton.CheckedChanged += new System.EventHandler(this.dimScreenOffRadioButton_CheckedChanged);
			honorPowerRequestsOnRadioButton.CheckedChanged += new System.EventHandler(this.honorPowerRequestsOnRadioButton_CheckedChanged);
			honorPowerRequestsOffRadioButton.CheckedChanged += new System.EventHandler(this.honorPowerRequestsOffRadioButton_CheckedChanged);
			honorUserInCallOnRadioButton.CheckedChanged += new System.EventHandler(this.honorUserInCallOnRadioButton_CheckedChanged);
			honorUserInCallOffRadioButton.CheckedChanged += new System.EventHandler(this.honorUserInCallOffRadioButton_CheckedChanged);
			lockAfterDimIntervalComboBox.SelectedIndexChanged += new System.EventHandler(this.lockIntervalComboBox_SelectedIndexChanged);
			displayOffOnRadioButton.CheckedChanged += new System.EventHandler(this.displayOffOnRadioButton_CheckedChanged);
			displayOffOffRadioButton.CheckedChanged += new System.EventHandler(this.displayOffOffRadioButton_CheckedChanged);

			woaCheckBox.CheckedChanged += new System.EventHandler(this.checkBoxWOA_CheckedChanged);
			woaExternalMonitorOffRadioButton.CheckedChanged += new System.EventHandler(this.nearExternalMonitorOffRadioButton_CheckedChanged);
			woaExternalMonitorOnRadioButton.CheckedChanged += new System.EventHandler(this.nearExternalMonitorOnRadioButton_CheckedChanged);
			woaEnableOnBatteryTrueRadioButton.CheckedChanged += new System.EventHandler(this.woaEnableOnBatteryTrue_radio_CheckedChanged);
			woaEnableOnBatteryFalseRadioButton.CheckedChanged += new System.EventHandler(this.woaEnableOnBatteryFalse_radio_CheckedChanged);
			woaBatteryRemainingComboBox.SelectedIndexChanged += new System.EventHandler(this.woaBatteryComboBox_SelectedIndexChanged);

			nlopCheckBox.CheckedChanged += new System.EventHandler(this.checkBoxNlop_CheckedChanged);
			nlopExternalMonitorOffRadioButton.CheckedChanged += new System.EventHandler(this.nlopExternalMonitorOffRadioButton_CheckedChanged);
			nlopExternalMonitorOnRadioButton.CheckedChanged += new System.EventHandler(this.nlopExternalMonitorOnRadioButton_CheckedChanged);
			nlopEnableOnBatteryTrueRadioButton.CheckedChanged += new System.EventHandler(this.nlopEnableOnBatteryTrue_radio_CheckedChanged);
			nlopEnableOnBatteryFalseRadioButton.CheckedChanged += new System.EventHandler(this.nlopEnableOnBatteryFalse_radio_CheckedChanged);
			nlopBatteryRemainingComboBox.SelectedIndexChanged += new System.EventHandler(this.nlopBatteryComboBox_SelectedIndexChanged);
			comboBoxWaitTimer.SelectedIndexChanged += new System.EventHandler(this.waitIntervalComboBox_SelectedIndexChanged);
			nlopFailSafeTimeoutComboBox.SelectedIndexChanged += new System.EventHandler(this.failSafeTimeoutComboBox_SelectedIndexChanged);

			adCheckBox.CheckedChanged += new System.EventHandler(this.checkBoxAd_CheckedChanged);
			adNotPresentDimWaitTimeComboBox.SelectedIndexChanged += new System.EventHandler(this.adNotPresentDimWaitTimeComboBox_SelectedIndexChanged);
			adExternalMonitorOffRadioButton.CheckedChanged += new System.EventHandler(this.adExternalMonitorOffRadioButton_CheckedChanged);
			adExternalMonitorOnRadioButton.CheckedChanged += new System.EventHandler(this.adExternalMonitorOnRadioButton_CheckedChanged);
			adHonorPowerRequestOnRadioButton.CheckedChanged += new System.EventHandler(this.adHonorPowerRequestOnRadioButton_CheckedChanged);
			adHonorPowerRequestOffRadioButton.CheckedChanged += new System.EventHandler(this.adHonorPowerRequestOffRadioButton_CheckedChanged);
			honorUserInCallADRadioButtonOn.CheckedChanged += new System.EventHandler(this.honorUserInCallADRadioButtonOn_CheckedChanged);
			honorUserInCallADRadioButtonOff.CheckedChanged += new System.EventHandler(this.honorUserInCallADRadioButtonOff_CheckedChanged);
			mispredictionWithFaceDetectionCheckbox.CheckedChanged += new System.EventHandler(this.mispredictionWithFaceDetectionCheckbox_CheckedChanged);
			mispredictionTimeWindowCombobox.SelectedIndexChanged += new System.EventHandler(this.mispredictionTimeWindowCombobox_SelectedIndexChanged);
			MP1DimWaitComboBox.SelectedIndexChanged += new System.EventHandler(this.MP1DimWaitComboBox_SelectedIndexChanged);
			MP2DimWaitComboBox.SelectedIndexChanged += new System.EventHandler(this.MP2DimWaitComboBox_SelectedIndexChanged);
			MP3DimWaitComboBox.SelectedIndexChanged += new System.EventHandler(this.MP3DimWaitComboBox_SelectedIndexChanged);
			MP4DimWaitComboBox.SelectedIndexChanged += new System.EventHandler(this.MP4DimWaitComboBox_SelectedIndexChanged);
			adDimTargetComboBox.SelectedIndexChanged += new System.EventHandler(this.adDimTargetComboBox_SelectedIndexChanged);
			adDimmingIntervalComboBox.SelectedIndexChanged += new System.EventHandler(this.adDimmingIntervalComboBox_SelectedIndexChanged);
			adDisengagedDimWaitTimeComboBox.SelectedIndexChanged += new System.EventHandler(this.adDisengagedDimWaitTimeComboBox_SelectedIndexChanged);

			onlookerDetectionCheckBox.CheckedChanged += new System.EventHandler(this.onlookerDetectionCheckBox_CheckedChanged);
			onlookerDetectionExternalMonitorOnRadioButton.CheckedChanged += new System.EventHandler(this.onlookerDetectionExternalMonitorOnRadioButton_CheckedChanged);
			onlookerDetectionExternalMonitorOffRadioButton.CheckedChanged += new System.EventHandler(this.onlookerDetectionExternalMonitorOffRadioButton_CheckedChanged);
			onlookerDetectionDelayIntervalComboBox.SelectedIndexChanged += new System.EventHandler(this.onlookerDetectionDelayIntervalComboBox_SelectedIndexChanged);
			onlookerDetectionDimWhenDetectedOnRadioButton.CheckedChanged += new System.EventHandler(this.onlookerDetectionDimWhenDetectedOnRadioButton_CheckedChanged);
			onlookerDetectionDimWhenDetectedOffRadioButton.CheckedChanged += new System.EventHandler(this.onlookerDetectionDimWhenDetectedOffRadioButton_CheckedChanged);
			onlookerDetectionDimmingTimeComboBox.SelectedIndexChanged += new System.EventHandler(this.onlookerDetectionDimmingTimeComboBox_SelectedIndexChanged);
			onlookerDetectionDimTargetComboBox.SelectedIndexChanged += new System.EventHandler(this.onlookerDetectionDimTargetComboBox_SelectedIndexChanged);
			onlookerDetectionEnableOnLowBatteryOnRadioButton.CheckedChanged += new System.EventHandler(this.onlookerDetectionEnableOnLowBatteryOnRadioButton_CheckedChanged);
			onlookerDetectionEnableOnLowBatteryOffRadioButton.CheckedChanged += new System.EventHandler(this.onlookerDetectionEnableOnLowBatteryOffRadioButton_CheckedChanged);
			onlookerDetectionLowBatteryLimitComboBox.SelectedIndexChanged += new System.EventHandler(this.onlookerDetectionLowBatteryLimitComboBox_SelectedIndexChanged);
			onlookerDetectionAbsenceIntervalComboBox.SelectedIndexChanged += new System.EventHandler(this.onlookerDetectionAbsenceIntervalComboBox_SelectedIndexChanged);

			privacyCheckbox.CheckedChanged += new System.EventHandler(this.privacyCheckBox_CheckedChanged);
			appNameComboBox.SelectedIndexChanged += new System.EventHandler(this.appNameComboBox_SelectedIndexChanged);
			appNameComboBox.SelectedIndexChanged += new System.EventHandler(this.appNameComboBox_SelectedIndexChanged);
			monitorAppCheckbox.CheckedChanged += new System.EventHandler(this.monitorAppCheckbox_CheckedChanged);
			enabledEventsListBox.SelectedIndexChanged += new System.EventHandler(this.enabledEventsListBox_SelectedIndexChanged);
		}

		private async void HandleSetOptionExceptions(Task setTask)
		{
			try
			{
				await setTask;
			}
			catch (ErrorException ee)
			{
				logger.Debug(ee.ErrorObject.Description);
				MessageBox.Show(ee.ErrorObject.Description, ee.ErrorType.ToString(), MessageBoxButtons.OK, MessageBoxIcon.Error);
			}
			catch (AggregateException ae)
			{
				foreach (var ie in ae.InnerExceptions)
				{
					ErrorException ee = (ErrorException)ie;
					logger.Debug(ee.ErrorObject.Description);
					MessageBox.Show(ee.ErrorObject.Description, ee.ErrorType.ToString(), MessageBoxButtons.OK, MessageBoxIcon.Error);
				}
			}
			catch (Exception ex)
			{
				logger.Debug(ex.Message);
			}
		}

		private void RemoveControlEventHandlers()
		{
			walCheckBox.CheckedChanged -= new System.EventHandler(this.checkBoxWAL_CheckedChanged);
			walExternalMonitorOnRadioButton.CheckedChanged -= new System.EventHandler(this.awayExternalMonitorOnRadioButton_CheckedChanged);
			walExternalMonitorOffRadioButton.CheckedChanged -= new System.EventHandler(this.awayExternalMonitorOffRadioButton_CheckedChanged);
			preDimIntervalComboBox.SelectedIndexChanged -= new System.EventHandler(this.preDimIntervalComboBox_SelectedIndexChanged);
			notPresentDimmingTimeComboBox.SelectedIndexChanged -= new System.EventHandler(this.notPresentDimmingTimeComboBox_SelectedIndexChanged);
			notPresentDimTargetComboBox.SelectedIndexChanged -= new System.EventHandler(this.notPresentDimTargetComboBox_SelectedIndexChanged);
			userPresentWaitIntervalComboBox.SelectedIndexChanged -= new System.EventHandler(this.userPresentWaitIntervalComboBox_SelectedIndexChanged);
			dimScreenOnRadioButton.CheckedChanged -= new System.EventHandler(this.dimScreenOnRadioButton_CheckedChanged);
			dimScreenOffRadioButton.CheckedChanged -= new System.EventHandler(this.dimScreenOffRadioButton_CheckedChanged);
			honorPowerRequestsOnRadioButton.CheckedChanged -= new System.EventHandler(this.honorPowerRequestsOnRadioButton_CheckedChanged);
			honorPowerRequestsOffRadioButton.CheckedChanged -= new System.EventHandler(this.honorPowerRequestsOffRadioButton_CheckedChanged);
			honorUserInCallOnRadioButton.CheckedChanged -= new System.EventHandler(this.honorUserInCallOnRadioButton_CheckedChanged);
			honorUserInCallOffRadioButton.CheckedChanged -= new System.EventHandler(this.honorUserInCallOffRadioButton_CheckedChanged);
			lockAfterDimIntervalComboBox.SelectedIndexChanged -= new System.EventHandler(this.lockIntervalComboBox_SelectedIndexChanged);
			displayOffOnRadioButton.CheckedChanged -= new System.EventHandler(this.displayOffOnRadioButton_CheckedChanged);
			displayOffOffRadioButton.CheckedChanged -= new System.EventHandler(this.displayOffOffRadioButton_CheckedChanged);

			woaCheckBox.CheckedChanged -= new System.EventHandler(this.checkBoxWOA_CheckedChanged);
			woaExternalMonitorOffRadioButton.CheckedChanged -= new System.EventHandler(this.nearExternalMonitorOffRadioButton_CheckedChanged);
			woaExternalMonitorOnRadioButton.CheckedChanged -= new System.EventHandler(this.nearExternalMonitorOnRadioButton_CheckedChanged);
			woaEnableOnBatteryTrueRadioButton.CheckedChanged -= new System.EventHandler(this.woaEnableOnBatteryTrue_radio_CheckedChanged);
			woaEnableOnBatteryFalseRadioButton.CheckedChanged -= new System.EventHandler(this.woaEnableOnBatteryFalse_radio_CheckedChanged);
			woaBatteryRemainingComboBox.SelectedIndexChanged -= new System.EventHandler(this.woaBatteryComboBox_SelectedIndexChanged);

			nlopCheckBox.CheckedChanged -= new System.EventHandler(this.checkBoxNlop_CheckedChanged);
			nlopExternalMonitorOffRadioButton.CheckedChanged -= new System.EventHandler(this.nlopExternalMonitorOffRadioButton_CheckedChanged);
			nlopExternalMonitorOnRadioButton.CheckedChanged -= new System.EventHandler(this.nlopExternalMonitorOnRadioButton_CheckedChanged);
			nlopEnableOnBatteryTrueRadioButton.CheckedChanged -= new System.EventHandler(this.nlopEnableOnBatteryTrue_radio_CheckedChanged);
			nlopEnableOnBatteryFalseRadioButton.CheckedChanged -= new System.EventHandler(this.nlopEnableOnBatteryFalse_radio_CheckedChanged);
			nlopBatteryRemainingComboBox.SelectedIndexChanged -= new System.EventHandler(this.nlopBatteryComboBox_SelectedIndexChanged);
			comboBoxWaitTimer.SelectedIndexChanged -= new System.EventHandler(this.waitIntervalComboBox_SelectedIndexChanged);
			nlopFailSafeTimeoutComboBox.SelectedIndexChanged -= new System.EventHandler(this.failSafeTimeoutComboBox_SelectedIndexChanged);

			adCheckBox.CheckedChanged -= new System.EventHandler(this.checkBoxAd_CheckedChanged);
			adNotPresentDimWaitTimeComboBox.SelectedIndexChanged -= new System.EventHandler(this.adNotPresentDimWaitTimeComboBox_SelectedIndexChanged);
			adExternalMonitorOffRadioButton.CheckedChanged -= new System.EventHandler(this.adExternalMonitorOffRadioButton_CheckedChanged);
			adExternalMonitorOnRadioButton.CheckedChanged -= new System.EventHandler(this.adExternalMonitorOnRadioButton_CheckedChanged);
			adHonorPowerRequestOnRadioButton.CheckedChanged -= new System.EventHandler(this.adHonorPowerRequestOnRadioButton_CheckedChanged);
			adHonorPowerRequestOffRadioButton.CheckedChanged -= new System.EventHandler(this.adHonorPowerRequestOffRadioButton_CheckedChanged);
			honorUserInCallADRadioButtonOn.CheckedChanged -= new System.EventHandler(this.honorUserInCallADRadioButtonOn_CheckedChanged);
			honorUserInCallADRadioButtonOff.CheckedChanged -= new System.EventHandler(this.honorUserInCallADRadioButtonOff_CheckedChanged);
			mispredictionWithFaceDetectionCheckbox.CheckedChanged -= new System.EventHandler(this.mispredictionWithFaceDetectionCheckbox_CheckedChanged);
			mispredictionTimeWindowCombobox.SelectedIndexChanged -= new System.EventHandler(this.mispredictionTimeWindowCombobox_SelectedIndexChanged);
			MP1DimWaitComboBox.SelectedIndexChanged -= new System.EventHandler(this.MP1DimWaitComboBox_SelectedIndexChanged);
			MP2DimWaitComboBox.SelectedIndexChanged -= new System.EventHandler(this.MP2DimWaitComboBox_SelectedIndexChanged);
			MP3DimWaitComboBox.SelectedIndexChanged -= new System.EventHandler(this.MP3DimWaitComboBox_SelectedIndexChanged);
			MP4DimWaitComboBox.SelectedIndexChanged -= new System.EventHandler(this.MP4DimWaitComboBox_SelectedIndexChanged);
			adDimTargetComboBox.SelectedIndexChanged -= new System.EventHandler(this.adDimTargetComboBox_SelectedIndexChanged);
			adDimmingIntervalComboBox.SelectedIndexChanged -= new System.EventHandler(this.adDimmingIntervalComboBox_SelectedIndexChanged);
			adDisengagedDimWaitTimeComboBox.SelectedIndexChanged -= new System.EventHandler(this.adDisengagedDimWaitTimeComboBox_SelectedIndexChanged);

			onlookerDetectionCheckBox.CheckedChanged -= new System.EventHandler(this.onlookerDetectionCheckBox_CheckedChanged);
			onlookerDetectionExternalMonitorOnRadioButton.CheckedChanged -= new System.EventHandler(this.onlookerDetectionExternalMonitorOnRadioButton_CheckedChanged);
			onlookerDetectionExternalMonitorOffRadioButton.CheckedChanged -= new System.EventHandler(this.onlookerDetectionExternalMonitorOffRadioButton_CheckedChanged);
			onlookerDetectionDelayIntervalComboBox.SelectedIndexChanged -= new System.EventHandler(this.onlookerDetectionDelayIntervalComboBox_SelectedIndexChanged);
			onlookerDetectionDimWhenDetectedOnRadioButton.CheckedChanged -= new System.EventHandler(this.onlookerDetectionDimWhenDetectedOnRadioButton_CheckedChanged);
			onlookerDetectionDimWhenDetectedOffRadioButton.CheckedChanged -= new System.EventHandler(this.onlookerDetectionDimWhenDetectedOffRadioButton_CheckedChanged);
			onlookerDetectionDimmingTimeComboBox.SelectedIndexChanged -= new System.EventHandler(this.onlookerDetectionDimmingTimeComboBox_SelectedIndexChanged);
			onlookerDetectionDimTargetComboBox.SelectedIndexChanged -= new System.EventHandler(this.onlookerDetectionDimTargetComboBox_SelectedIndexChanged);
			onlookerDetectionEnableOnLowBatteryOnRadioButton.CheckedChanged -= new System.EventHandler(this.onlookerDetectionEnableOnLowBatteryOnRadioButton_CheckedChanged);
			onlookerDetectionEnableOnLowBatteryOffRadioButton.CheckedChanged -= new System.EventHandler(this.onlookerDetectionEnableOnLowBatteryOffRadioButton_CheckedChanged);
			onlookerDetectionLowBatteryLimitComboBox.SelectedIndexChanged -= new System.EventHandler(this.onlookerDetectionLowBatteryLimitComboBox_SelectedIndexChanged);
			onlookerDetectionAbsenceIntervalComboBox.SelectedIndexChanged -= new System.EventHandler(this.onlookerDetectionAbsenceIntervalComboBox_SelectedIndexChanged);

			privacyCheckbox.CheckedChanged -= new System.EventHandler(this.privacyCheckBox_CheckedChanged);
			appNameComboBox.SelectedIndexChanged -= new System.EventHandler(this.appNameComboBox_SelectedIndexChanged);
			monitorAppCheckbox.CheckedChanged -= new System.EventHandler(this.monitorAppCheckbox_CheckedChanged);
			enabledEventsListBox.SelectedIndexChanged -= new System.EventHandler(this.enabledEventsListBox_SelectedIndexChanged);
		}

		private void LoadControlPresets()
		{
			preDimIntervalComboBox.Items.AddRange(Array.ConvertAll(preDimIntervalSeconds, x => x.ToString() + " Seconds"));
			notPresentDimmingTimeComboBox.Items.AddRange(Array.ConvertAll(dimmingIntervalSeconds, x => x.ToString() + " Seconds"));
			notPresentDimTargetComboBox.Items.AddRange(Array.ConvertAll(dimTarget, x => x.ToString() + " %"));
			userPresentWaitIntervalComboBox.Items.AddRange(Array.ConvertAll(userPresentWaitIntervalsSeconds, x => x.ToString() + " Seconds"));
			lockAfterDimIntervalComboBox.Items.AddRange(Array.ConvertAll(lockAfterDimIntervalsSeconds, x => x.ToString() + " Seconds"));

			woaBatteryRemainingComboBox.Items.AddRange(Array.ConvertAll(batteryRemainingPercentage, x => x.ToString() + " %"));

			nlopBatteryRemainingComboBox.Items.AddRange(Array.ConvertAll(batteryRemainingPercentage, x => x.ToString() + " %"));
			comboBoxWaitTimer.Items.AddRange(Array.ConvertAll(preDimIntervalSeconds, x => x.ToString() + " Seconds"));
			nlopFailSafeTimeoutComboBox.Items.AddRange(Array.ConvertAll(failSafeIntervalSeconds, x => x.ToString() + " Seconds"));

			adNotPresentDimWaitTimeComboBox.Items.AddRange(Array.ConvertAll(preDimIntervalSeconds, x => x.ToString() + " Seconds"));
			MP1DimWaitComboBox.Items.AddRange(Array.ConvertAll(mispredictionDimWaitSeconds, x => x.ToString() + " Seconds"));
			MP2DimWaitComboBox.Items.AddRange(Array.ConvertAll(mispredictionDimWaitSeconds, x => x.ToString() + " Seconds"));
			MP3DimWaitComboBox.Items.AddRange(Array.ConvertAll(mispredictionDimWaitSeconds, x => x.ToString() + " Seconds"));
			MP4DimWaitComboBox.Items.AddRange(Array.ConvertAll(mispredictionDimWaitSeconds, x => x.ToString() + " Seconds"));
			mispredictionTimeWindowCombobox.Items.AddRange(Array.ConvertAll(mispredictionTimeWindowSeconds, x => x.ToString() + " Seconds"));
			adDimTargetComboBox.Items.AddRange(Array.ConvertAll(dimTarget, x => x.ToString() + " %"));
			adDimmingIntervalComboBox.Items.AddRange(Array.ConvertAll(dimmingIntervalSeconds, x => x.ToString() + " Seconds"));
			adDisengagedDimWaitTimeComboBox.Items.AddRange(Array.ConvertAll(preDimIntervalSeconds, x => x.ToString() + " Seconds"));

			onlookerDetectionDelayIntervalComboBox.Items.AddRange(Array.ConvertAll(preDimIntervalSeconds, x => x.ToString() + " Seconds"));
			onlookerDetectionDimmingTimeComboBox.Items.AddRange(Array.ConvertAll(dimmingIntervalSeconds, x => x.ToString() + " Seconds"));
			onlookerDetectionDimTargetComboBox.Items.AddRange(Array.ConvertAll(dimTarget, x => x.ToString() + " %"));
			onlookerDetectionLowBatteryLimitComboBox.Items.AddRange(Array.ConvertAll(batteryRemainingPercentage, x => x.ToString() + " %"));
			onlookerDetectionAbsenceIntervalComboBox.Items.AddRange(Array.ConvertAll(detectionAbsenceIntervalSeconds, x => x.ToString() + " Seconds"));

			appNameComboBox.Items.AddRange(appNames);
			enabledEventsListBox.Items.AddRange(eventNames);
		}

		internal void MakeIntervalControlsVisible(bool visible)
		{
			//labelMinLockInterval.Visible = visible;
			//todo:add new controls
		}

		#endregion GUI initialization

		#region helpers

		internal void StartHumanPresenceUpdateTask()
		{
			if (humanPresenceBackgroundTask.IsCompleted && this.HumanPresencePollEnabled)
			{
				humanPresenceTaskTokenSource.Cancel();
				humanPresenceTaskTokenSource = new CancellationTokenSource();
				humanPresenceBackgroundTask = HumanPresenceUpdateBackgroundTask(humanPresenceTaskTokenSource.Token);
			}
		}

		internal void StopHumanPresenceUpdateTask()
		{
			humanPresenceTaskTokenSource.Cancel();
		}

		internal async Task HumanPresenceUpdateBackgroundTask(CancellationToken stoppingToken)
		{
			try
			{
				while (!stoppingToken.IsCancellationRequested)
				{
					UpdatePresenceStatus();
					await Task.Delay(HUMAN_PRESENCE_UPDATE_INTERVAL, stoppingToken);
				}
			}
			catch (OperationCanceledException)
			{
				return;
			}
			catch (Exception ex)
			{
				logger.Error($"Human Presence background task stopped due to error: {ex}");
			}
			finally
			{
				logger.Debug("Human Presence background task stopped.");
			}
		}

		internal void UpdateWALControlsVisibility(bool visible)
		{
			panelWal.Enabled = visible;
			panelWal.Visible = visible;
		}
		internal void UpdateWOAControlsVisibility(bool visible)
		{
			panelWoa.Enabled = visible;
			panelWoa.Visible = visible;
		}

		internal void UpdateNLoPControlsVisibility(bool visible)
		{
			panelNlop.Enabled = visible;
			panelNlop.Visible = visible;
		}

		internal void UpdateAdControlsVisibility(bool visible)
		{
			panelAd.Enabled = visible;
			panelAd.Visible = visible;
		}

		internal void UpdatePrivacyControlsVisibility(bool visible)
		{
			privacyPanel.Enabled = visible;
		}

		internal void UpdateOnlookerDetectionControlsVisibility(bool visible)
		{
			onlookerDetectionSettingsPanel.Enabled = visible;
			onlookerDetectionSettingsPanel.Visible = visible;
		}

		internal void UpdateAppFeatureControlsVisibility(bool visible)
		{
			appFeaturesPanel.Enabled = visible;
			appFeaturesPanel.Visible = visible;
			appEnabledPanel.Visible = visible;
			extendedCapabilitiesPanel.Visible = visible;
			helperActivePanel.Visible = visible;

			// Privacy is always visible, but test mode will override its enabled setting
			privacyCheckbox.Enabled = visible || this.visionCapable;
		}

		private string CollectionToString(System.Collections.Specialized.StringCollection collection)
		{
			StringBuilder sb = new StringBuilder();
			foreach (string str in collection)
			{
				sb.Append(str);
				sb.Append(",");
			}
			return sb.ToString();
		}

		private void SetAppName()
		{
			string appName = "Intel.ContextSensingSampleApp";
			Task setTask = _commClient.SetOption(FeatureType.APPLICATION, FeatureProperty.AppName, appName);
			HandleSetOptionExceptions(setTask);
		}

		private void SubscribeToCapabilityEvents()
		{
			var subscribedEvents = GetSubscribedEvents(enabledEventsListBox.SelectedItems);

			subscribedEvents.Add(EventTypeMap.EventSubscriptionNames[EventType.EVENT_CST_CAPABILITY_CHANGED]);
			subscribedEvents.Add(EventTypeMap.EventSubscriptionNames[EventType.EVENT_CST_EXTENDED_CAPABILITY_CHANGED]);
			subscribedEvents = subscribedEvents.Distinct().ToList();

			Task setTask = _commClient.SetOption(FeatureType.APPLICATION, FeatureProperty.EnabledEvents, subscribedEvents);
			HandleSetOptionExceptions(setTask);

			enabledEventsListBox.SelectedIndexChanged -= enabledEventsListBox_SelectedIndexChanged;
			UpdateEventListBox(subscribedEvents);
			enabledEventsListBox.SelectedIndexChanged += enabledEventsListBox_SelectedIndexChanged;
		}

		private List<string> GetSubscribedEvents(ListBox.SelectedObjectCollection selectedEvents)
		{
			var subscribedEventsList = new List<string>();
			foreach (string eventName in selectedEvents)
			{
				if (eventNames.Contains(eventName))
				{
					// sanity check to make sure we don't send bad data to the service
					subscribedEventsList.Add(eventName);
				}
			}
			return subscribedEventsList;
		}

		#endregion helpers

		private void awayExternalMonitorOnRadioButton_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.LOCK, FeatureProperty.EnableWithExtMonitor, true, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void awayExternalMonitorOffRadioButton_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.LOCK, FeatureProperty.EnableWithExtMonitor, false, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void preDimIntervalComboBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ComboBox)
			{
				if (((ComboBox)sender).SelectedIndex < preDimIntervalSeconds.Length)
				{
					Task setTask = _commClient.SetOption(FeatureType.LOCK, FeatureProperty.PreDimInterval, preDimIntervalSeconds[((ComboBox)sender).SelectedIndex] * 1000, globalMode);
					HandleSetOptionExceptions(setTask);
				}
			}
		}

		private void notPresentDimmingTimeComboBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ComboBox)
			{
				if (((ComboBox)sender).SelectedIndex < dimmingIntervalSeconds.Length)
				{
					Task setTask = _commClient.SetOption(FeatureType.LOCK, FeatureProperty.DimmingInterval, dimmingIntervalSeconds[((ComboBox)sender).SelectedIndex] * 1000, globalMode);
					HandleSetOptionExceptions(setTask);
				}
			}
		}

		private void notPresentDimTargetComboBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ComboBox)
			{
				if (((ComboBox)sender).SelectedIndex < dimTarget.Length)
				{
					Task setTask = _commClient.SetOption(FeatureType.LOCK, FeatureProperty.NotPresentDimTarget, dimTarget[((ComboBox)sender).SelectedIndex], globalMode);
					HandleSetOptionExceptions(setTask);
				}
			}
		}

		private void userPresentWaitIntervalComboBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < userPresentWaitIntervalsSeconds.Length)
			{
				Task setTask = _commClient.SetOption(FeatureType.LOCK, FeatureProperty.WaitInterval, userPresentWaitIntervalsSeconds[((ComboBox)sender).SelectedIndex] * 1000, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void dimScreenOnRadioButton_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.LOCK, FeatureProperty.DimScreen, true, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void dimScreenOffRadioButton_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.LOCK, FeatureProperty.DimScreen, false, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void honorPowerRequestsOnRadioButton_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.LOCK, FeatureProperty.PowerRequest, true, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void honorPowerRequestsOffRadioButton_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.LOCK, FeatureProperty.PowerRequest, false, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void honorUserInCallOnRadioButton_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.LOCK, FeatureProperty.EnableAudioMode, true, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void honorUserInCallOffRadioButton_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.LOCK, FeatureProperty.EnableAudioMode, false, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void lockIntervalComboBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < lockAfterDimIntervalsSeconds.Length)
			{
				Task setTask = _commClient.SetOption(FeatureType.LOCK, FeatureProperty.LockAfterDimInterval, lockAfterDimIntervalsSeconds[((ComboBox)sender).SelectedIndex] * 1000, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void displayOffOnRadioButton_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.LOCK, FeatureProperty.DisplayOffAfterLock, true, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void displayOffOffRadioButton_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.LOCK, FeatureProperty.DisplayOffAfterLock, false, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void nearExternalMonitorOnRadioButton_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.WAKE, FeatureProperty.EnableWithExtMonitor, true, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void nearExternalMonitorOffRadioButton_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.WAKE, FeatureProperty.EnableWithExtMonitor, false, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void woaEnableOnBatteryTrue_radio_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.WAKE, FeatureProperty.WakeOnLowBattery, true, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void woaEnableOnBatteryFalse_radio_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.WAKE, FeatureProperty.WakeOnLowBattery, false, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void woaBatteryComboBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ComboBox)
			{
				if (((ComboBox)sender).SelectedIndex < batteryRemainingPercentage.Length)
				{
					Task setTask = _commClient.SetOption(FeatureType.WAKE, FeatureProperty.LowBatteryRemainingPercentageLimit, batteryRemainingPercentage[((ComboBox)sender).SelectedIndex], globalMode);
					HandleSetOptionExceptions(setTask);
				}
			}
		}

		private void nlopExternalMonitorOffRadioButton_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.NOLOCKONPRESENCE, FeatureProperty.EnableWithExtMonitor, false, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void nlopExternalMonitorOnRadioButton_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.NOLOCKONPRESENCE, FeatureProperty.EnableWithExtMonitor, true, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void nlopEnableOnBatteryTrue_radio_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.NOLOCKONPRESENCE, FeatureProperty.EnableOnBattery, true, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void nlopEnableOnBatteryFalse_radio_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.NOLOCKONPRESENCE, FeatureProperty.EnableOnBattery, false, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void nlopBatteryComboBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ComboBox)
			{
				if (((ComboBox)sender).SelectedIndex < batteryRemainingPercentage.Length)
				{
					Task setTask = _commClient.SetOption(FeatureType.NOLOCKONPRESENCE, FeatureProperty.LowBatteryRemainingPercentageLimit, batteryRemainingPercentage[((ComboBox)sender).SelectedIndex], globalMode);
					HandleSetOptionExceptions(setTask);
				}
			}
		}

		private void waitIntervalComboBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < preDimIntervalSeconds.Length)
			{
				Task setTask = _commClient.SetOption(FeatureType.NOLOCKONPRESENCE, FeatureProperty.WaitInterval, preDimIntervalSeconds[((ComboBox)sender).SelectedIndex] * 1000, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void failSafeTimeoutComboBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < preDimIntervalSeconds.Length)
			{
				Task setTask = _commClient.SetOption(FeatureType.NOLOCKONPRESENCE, FeatureProperty.FailSafeTimeout, failSafeIntervalSeconds[((ComboBox)sender).SelectedIndex] * 1000, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void adExternalMonitorOnRadioButton_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.EnableWithExtMonitor, true, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void adExternalMonitorOffRadioButton_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.EnableWithExtMonitor, false, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void adNotPresentDimWaitTimeComboBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < preDimIntervalSeconds.Length)
			{
				Task setTask = _commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.NotPresentDimWaitTime, preDimIntervalSeconds[((ComboBox)sender).SelectedIndex] * 1000, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void mispredictionWithFaceDetectionCheckbox_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is CheckBox)
			{
				if (((CheckBox)sender).CheckState == CheckState.Checked)
				{
					Task setTask = _commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.MispredictionFaceDetection, true, globalMode);
					HandleSetOptionExceptions(setTask);
				}
				else
				{
					Task setTask = _commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.MispredictionFaceDetection, false, globalMode);
					HandleSetOptionExceptions(setTask);
				}
			}
		}

		private void honorUserInCallADRadioButtonOff_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.EnableAudioMode, false, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void honorUserInCallADRadioButtonOn_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.EnableAudioMode, true, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}
		private void mispredictionTimeWindowCombobox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < mispredictionTimeWindowSeconds.Length)
			{
				Task setTask = _commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.MispredictionTimeWindow, mispredictionTimeWindowSeconds[((ComboBox)sender).SelectedIndex] * 1000, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void MP1DimWaitComboBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < mispredictionDimWaitSeconds.Length)
			{
				Task setTask = _commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.Misprediction1DimWaitTime, mispredictionDimWaitSeconds[((ComboBox)sender).SelectedIndex] * 1000, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void MP2DimWaitComboBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < mispredictionDimWaitSeconds.Length)
			{
				Task setTask = _commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.Misprediction2DimWaitTime, mispredictionDimWaitSeconds[((ComboBox)sender).SelectedIndex] * 1000, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void MP3DimWaitComboBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < mispredictionDimWaitSeconds.Length)
			{
				Task setTask = _commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.Misprediction3DimWaitTime, mispredictionDimWaitSeconds[((ComboBox)sender).SelectedIndex] * 1000, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void MP4DimWaitComboBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < mispredictionDimWaitSeconds.Length)
			{
				Task setTask = _commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.Misprediction4DimWaitTime, mispredictionDimWaitSeconds[((ComboBox)sender).SelectedIndex] * 1000, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void adDimTargetComboBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ComboBox)
			{
				if (((ComboBox)sender).SelectedIndex < dimTarget.Length)
				{
					Task setTask = _commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.DimTarget, dimTarget[((ComboBox)sender).SelectedIndex], globalMode);
					HandleSetOptionExceptions(setTask);
				}
			}
		}

		private void adDimmingIntervalComboBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ComboBox)
			{
				if (((ComboBox)sender).SelectedIndex < dimmingIntervalSeconds.Length)
				{
					Task setTask = _commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.DimmingInterval, dimmingIntervalSeconds[((ComboBox)sender).SelectedIndex] * 1000, globalMode);
					HandleSetOptionExceptions(setTask);
				}
			}
		}

		private void adDisengagedDimWaitTimeComboBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ComboBox)
			{
				if (((ComboBox)sender).SelectedIndex < preDimIntervalSeconds.Length)
				{
					Task setTask = _commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.DisengagedDimWaitTime, preDimIntervalSeconds[((ComboBox)sender).SelectedIndex] * 1000, globalMode);
					HandleSetOptionExceptions(setTask);
				}
			}
		}

		private void onlookerDetectionDelayIntervalComboBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < preDimIntervalSeconds.Length)
			{
				Task setTask = _commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.DetectionDelayInterval, preDimIntervalSeconds[((ComboBox)sender).SelectedIndex] * 1000, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void onlookerDetectionDimmingTimeComboBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < dimmingIntervalSeconds.Length)
			{
				Task setTask = _commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.DimmingInterval, dimmingIntervalSeconds[((ComboBox)sender).SelectedIndex] * 1000, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void onlookerDetectionDimTargetComboBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < dimTarget.Length)
			{
				Task setTask = _commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.DimTarget, dimTarget[((ComboBox)sender).SelectedIndex], globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void onlookerDetectionLowBatteryLimitComboBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < batteryRemainingPercentage.Length)
			{
				Task setTask = _commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.LowBatteryRemainingPercentageLimit, batteryRemainingPercentage[((ComboBox)sender).SelectedIndex], globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void onlookerDetectionAbsenceIntervalComboBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ComboBox && ((ComboBox)sender).SelectedIndex < detectionAbsenceIntervalSeconds.Length)
			{
				Task setTask = _commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.OnlookerAbsenceWaitInterval, detectionAbsenceIntervalSeconds[((ComboBox)sender).SelectedIndex] * 1000, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void onlookerDetectionExternalMonitorOnRadioButton_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.EnableWithExtMonitor, true, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void onlookerDetectionExternalMonitorOffRadioButton_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.EnableWithExtMonitor, false, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void onlookerDetectionDimWhenDetectedOnRadioButton_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.DimWhenOnlookerDetected, true, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void onlookerDetectionDimWhenDetectedOffRadioButton_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.DimWhenOnlookerDetected, false, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void onlookerDetectionEnableOnLowBatteryOnRadioButton_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.EnableOnLowBattery, true, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void onlookerDetectionEnableOnLowBatteryOffRadioButton_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.ONLOOKERDETECTION, FeatureProperty.EnableOnLowBattery, false, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void appNameComboBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ComboBox)
			{
				if (((ComboBox)sender).SelectedIndex < appNames.Length)
				{
					Task setTask = _commClient.SetOption(FeatureType.APPLICATION, FeatureProperty.AppName, appNames[((ComboBox)sender).SelectedIndex], globalMode);
					HandleSetOptionExceptions(setTask);
				}
			}
		}

		private void globalModeCheckbox_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is CheckBox)
			{
				if (((CheckBox)sender).CheckState == CheckState.Checked)
				{
					globalMode = SettingsAccessType.GLOBAL;
					UpdateWOAControlsVisibility(true);
					UpdateWALControlsVisibility(true);
					UpdateNLoPControlsVisibility(true);
					UpdateAdControlsVisibility(true);
					UpdateOnlookerDetectionControlsVisibility(true);
					UpdatePrivacyControlsVisibility(true);
					// app panel visibility is controlled separately, even in global mode
					// see the "test mode" checkbox
				}
				else
				{
					globalMode = SettingsAccessType.LOCAL;
				}
				_ = RefreshFeaturesViaGetOptions();
			}
		}

		internal void UpdateControlSettings(FeatureSetting featureSetting)
		{
			if (featureSetting is LockFeatures)
			{
				UpdateControlSettings((LockFeatures)featureSetting);
			}
			else if (featureSetting is WakeFeatures)
			{
				UpdateControlSettings((WakeFeatures)featureSetting);
			}
			else if (featureSetting is NoLockOnPresenceFeatures)
			{
				UpdateControlSettings((NoLockOnPresenceFeatures)featureSetting);
			}
			else if (featureSetting is AdaptiveDimmingFeatures)
			{
				UpdateControlSettings((AdaptiveDimmingFeatures)featureSetting);
			}
			else if (featureSetting is OnlookerDetectionFeatures)
			{
				UpdateControlSettings((OnlookerDetectionFeatures)featureSetting);
			}
			else if (featureSetting is AppFeatures)
			{
				UpdateControlSettings((AppFeatures)featureSetting);
			}

			globalModeCheckbox.Checked = (globalMode == SettingsAccessType.GLOBAL);
		}

		string convertToPascalCase(string str)
		{
			return str.First().ToString().ToUpper() + str.Substring(1);
		}

		internal void UpdateControlSettings(LockFeatures lockFeatures)
		{
			RemoveControlEventHandlers();
			walCheckBox.Checked = lockFeatures.WALEnabled;
			walExternalMonitorOnRadioButton.Checked = lockFeatures.EnableLockWithExternalMonitor;
			walExternalMonitorOffRadioButton.Checked = !lockFeatures.EnableLockWithExternalMonitor;
			preDimIntervalComboBox.SelectedIndex = Array.IndexOf(preDimIntervalSeconds, lockFeatures.PreDimInterval / 1000);
			notPresentDimmingTimeComboBox.SelectedIndex = Array.IndexOf(dimmingIntervalSeconds, lockFeatures.DimmingInterval / 1000);
			notPresentDimTargetComboBox.SelectedIndex = Array.IndexOf(dimTarget, lockFeatures.NotPresentDimTarget);
			userPresentWaitIntervalComboBox.SelectedIndex = Array.IndexOf(userPresentWaitIntervalsSeconds, lockFeatures.WaitInterval / 1000);
			dimScreenOnRadioButton.Checked = lockFeatures.DimScreen;
			dimScreenOffRadioButton.Checked = !lockFeatures.DimScreen;
			honorPowerRequestsOnRadioButton.Checked = lockFeatures.PowerRequest;
			honorPowerRequestsOffRadioButton.Checked = !lockFeatures.PowerRequest;
			honorUserInCallOnRadioButton.Checked = lockFeatures.EnableAudioMode;
			honorUserInCallOffRadioButton.Checked = !lockFeatures.EnableAudioMode;
			lockAfterDimIntervalComboBox.SelectedIndex = Array.IndexOf(lockAfterDimIntervalsSeconds, lockFeatures.LockAfterDimInterval / 1000);
			displayOffOnRadioButton.Checked = lockFeatures.DisplayOffAfterLock;
			displayOffOffRadioButton.Checked = !lockFeatures.DisplayOffAfterLock;

			AddControlEventHandlers();
			UpdateWALControlsVisibility(walCheckBox.Checked);
		}
		internal void UpdateControlSettings(WakeFeatures wakeFeatures)
		{
			RemoveControlEventHandlers();
			woaCheckBox.Checked = wakeFeatures.WOAEnabled;
			woaExternalMonitorOnRadioButton.Checked = wakeFeatures.EnableWakeWithExternalMonitor;
			woaExternalMonitorOffRadioButton.Checked = !wakeFeatures.EnableWakeWithExternalMonitor;
			woaEnableOnBatteryTrueRadioButton.Checked = wakeFeatures.WakeOnLowBattery;
			woaEnableOnBatteryFalseRadioButton.Checked = !wakeFeatures.WakeOnLowBattery;
			woaBatteryRemainingComboBox.SelectedIndex = Array.IndexOf(batteryRemainingPercentage, wakeFeatures.LowBatteryRemainingPercentageLimit);

			AddControlEventHandlers();

			UpdateWOAControlsVisibility(woaCheckBox.Checked);
		}

		internal void UpdateControlSettings(NoLockOnPresenceFeatures nlopfeatures)
		{
			RemoveControlEventHandlers();
			nlopCheckBox.Checked = nlopfeatures.NLoPEnabled;
			nlopExternalMonitorOnRadioButton.Checked = nlopfeatures.EnableNLoPWithExternalMonitor;
			nlopExternalMonitorOffRadioButton.Checked = !nlopfeatures.EnableNLoPWithExternalMonitor;
			nlopEnableOnBatteryTrueRadioButton.Checked = nlopfeatures.EnableOnBattery;
			nlopEnableOnBatteryFalseRadioButton.Checked = !nlopfeatures.EnableOnBattery;
			nlopBatteryRemainingComboBox.SelectedIndex = Array.IndexOf(batteryRemainingPercentage, nlopfeatures.LowBatteryRemainingPercentageLimit);
			nlopBatteryRemainingComboBox.SelectedIndex = Array.IndexOf(batteryRemainingPercentage, nlopfeatures.LowBatteryRemainingPercentageLimit);
			comboBoxWaitTimer.SelectedIndex = Array.IndexOf(preDimIntervalSeconds, nlopfeatures.WaitInterval / 1000);
			nlopFailSafeTimeoutComboBox.SelectedIndex = Array.IndexOf(failSafeIntervalSeconds, nlopfeatures.FailSafeTimeout / 1000);
			UpdateNLoPControlsVisibility(nlopCheckBox.Checked);
			AddControlEventHandlers();
		}

		internal void UpdateControlSettings(AdaptiveDimmingFeatures adaptiveDimmingFeatures)
		{
			RemoveControlEventHandlers();
			adCheckBox.Checked = adaptiveDimmingFeatures.ADEnabled;
			adExternalMonitorOnRadioButton.Checked = adaptiveDimmingFeatures.EnableAdaptiveDimmingWithExternalMonitor;
			adExternalMonitorOffRadioButton.Checked = !adaptiveDimmingFeatures.EnableAdaptiveDimmingWithExternalMonitor;
			adHonorPowerRequestOnRadioButton.Checked = adaptiveDimmingFeatures.PowerRequest;
			adHonorPowerRequestOffRadioButton.Checked = !adaptiveDimmingFeatures.PowerRequest;
			honorUserInCallADRadioButtonOn.Checked = adaptiveDimmingFeatures.EnableAudioMode;
			honorUserInCallADRadioButtonOff.Checked = !adaptiveDimmingFeatures.EnableAudioMode;
			adNotPresentDimWaitTimeComboBox.SelectedIndex = Array.IndexOf(preDimIntervalSeconds, adaptiveDimmingFeatures.NotPresentDimWaitTime / 1000);
			mispredictionTimeWindowCombobox.SelectedIndex = Array.IndexOf(mispredictionTimeWindowSeconds, adaptiveDimmingFeatures.MispredictionTimeWindow / 1000);
			mispredictionWithFaceDetectionCheckbox.Checked = adaptiveDimmingFeatures.MispredictionFaceDetection;
			MP1DimWaitComboBox.SelectedIndex = Array.IndexOf(mispredictionDimWaitSeconds, adaptiveDimmingFeatures.Misprediction1DimWaitTime / 1000);
			MP2DimWaitComboBox.SelectedIndex = Array.IndexOf(mispredictionDimWaitSeconds, adaptiveDimmingFeatures.Misprediction2DimWaitTime / 1000);
			MP3DimWaitComboBox.SelectedIndex = Array.IndexOf(mispredictionDimWaitSeconds, adaptiveDimmingFeatures.Misprediction3DimWaitTime / 1000);
			MP4DimWaitComboBox.SelectedIndex = Array.IndexOf(mispredictionDimWaitSeconds, adaptiveDimmingFeatures.Misprediction4DimWaitTime / 1000);
			adDimTargetComboBox.SelectedIndex = Array.IndexOf(dimTarget, adaptiveDimmingFeatures.DimTarget);
			adDimmingIntervalComboBox.SelectedIndex = Array.IndexOf(dimmingIntervalSeconds, adaptiveDimmingFeatures.DimmingInterval / 1000);
			adDisengagedDimWaitTimeComboBox.SelectedIndex = Array.IndexOf(preDimIntervalSeconds, adaptiveDimmingFeatures.DisengagedDimWaitTime / 1000);
			AddControlEventHandlers();
			UpdateAdControlsVisibility(adCheckBox.Checked);
		}

		internal void UpdateControlSettings(AppFeatures appFeatures)
		{
			RemoveControlEventHandlers();
			privacyCheckbox.Checked = appFeatures.PrivacyEnabled;
			appNameComboBox.SelectedIndex = Array.IndexOf(appNames, appFeatures.AppName);
			monitorAppCheckbox.Checked = appFeatures.MonitorApp;
			UpdateEventListBox(appFeatures.EnabledEvents);
			AddControlEventHandlers();
		}

		private void UpdateEventListBox(List<string> eventList)
		{
			foreach (string eventName in this.eventNames)
			{
				int index = this.enabledEventsListBox.FindStringExact(eventName);
				if (index != ListBox.NoMatches)
				{
					this.enabledEventsListBox.SetSelected(index, eventList.Contains(eventName));
				}
			}
		}

		internal void UpdateControlSettings(OnlookerDetectionFeatures onlookerDetectionFeatures)
		{
			RemoveControlEventHandlers();
			onlookerDetectionCheckBox.Checked = onlookerDetectionFeatures.OnlookerDetectionEnabled;
			onlookerDetectionExternalMonitorOnRadioButton.Checked = onlookerDetectionFeatures.EnableOnlookerDetectionWithExternalMonitor;
			onlookerDetectionExternalMonitorOffRadioButton.Checked = !onlookerDetectionFeatures.EnableOnlookerDetectionWithExternalMonitor;
			onlookerDetectionDelayIntervalComboBox.SelectedIndex = Array.IndexOf(preDimIntervalSeconds, onlookerDetectionFeatures.DetectionDelayInterval / 1000);
			onlookerDetectionDimWhenDetectedOnRadioButton.Checked = onlookerDetectionFeatures.DimWhenOnlookerDetected;
			onlookerDetectionDimWhenDetectedOffRadioButton.Checked = !onlookerDetectionFeatures.DimWhenOnlookerDetected;
			onlookerDetectionDimmingTimeComboBox.SelectedIndex = Array.IndexOf(dimmingIntervalSeconds, onlookerDetectionFeatures.DimmingInterval / 1000);
			onlookerDetectionDimTargetComboBox.SelectedIndex = Array.IndexOf(dimTarget, onlookerDetectionFeatures.DimTarget);
			onlookerDetectionEnableOnLowBatteryOnRadioButton.Checked = onlookerDetectionFeatures.EnableOnLowBattery;
			onlookerDetectionEnableOnLowBatteryOffRadioButton.Checked = !onlookerDetectionFeatures.EnableOnLowBattery;
			onlookerDetectionLowBatteryLimitComboBox.SelectedIndex = Array.IndexOf(batteryRemainingPercentage, onlookerDetectionFeatures.LowBatteryRemainingPercentageLimit);
			onlookerDetectionAbsenceIntervalComboBox.SelectedIndex = Array.IndexOf(detectionAbsenceIntervalSeconds, onlookerDetectionFeatures.OnlookerAbsenceWaitInterval / 1000);

			AddControlEventHandlers();
			UpdateOnlookerDetectionControlsVisibility(onlookerDetectionCheckBox.Checked);
		}

		internal void ClearFeatureCheckboxes()
		{
			nlopCheckBox.CheckedChanged -= checkBoxNlop_CheckedChanged;
			nlopCheckBox.CheckState = CheckState.Unchecked;
			nlopCheckBox.CheckedChanged += checkBoxNlop_CheckedChanged;
			UpdateNLoPControlsVisibility(false);

			walCheckBox.CheckedChanged -= checkBoxWAL_CheckedChanged;
			walCheckBox.CheckState = CheckState.Unchecked;
			walCheckBox.CheckedChanged += checkBoxWAL_CheckedChanged;
			UpdateWALControlsVisibility(false);

			woaCheckBox.CheckedChanged -= checkBoxWOA_CheckedChanged;
			woaCheckBox.CheckState = CheckState.Unchecked;
			woaCheckBox.CheckedChanged += checkBoxWOA_CheckedChanged;
			UpdateWOAControlsVisibility(false);

			adCheckBox.CheckedChanged -= checkBoxAd_CheckedChanged;
			adCheckBox.CheckState = CheckState.Unchecked;
			adCheckBox.CheckedChanged += checkBoxAd_CheckedChanged;
			UpdateAdControlsVisibility(false);

			onlookerDetectionCheckBox.CheckedChanged -= onlookerDetectionCheckBox_CheckedChanged;
			onlookerDetectionCheckBox.CheckState = CheckState.Unchecked;
			onlookerDetectionCheckBox.CheckedChanged += onlookerDetectionCheckBox_CheckedChanged;
			UpdateOnlookerDetectionControlsVisibility(false);

			privacyCheckbox.CheckedChanged -= privacyCheckBox_CheckedChanged;
			privacyCheckbox.CheckState = CheckState.Unchecked;
			privacyCheckbox.CheckedChanged += privacyCheckBox_CheckedChanged;
			UpdatePrivacyControlsVisibility(this.testModeCheckbox.Checked);

			monitorAppCheckbox.CheckedChanged -= monitorAppCheckbox_CheckedChanged;
			monitorAppCheckbox.CheckState = CheckState.Unchecked;
			monitorAppCheckbox.CheckedChanged += monitorAppCheckbox_CheckedChanged;
			UpdateAppFeatureControlsVisibility(this.testModeCheckbox.Checked);
		}

		internal void EnableFeatureCheckboxes(bool enabled = true)
		{
			nlopCheckBox.Enabled = enabled;
			walCheckBox.Enabled = enabled;
			woaCheckBox.Enabled = enabled;
			adCheckBox.Enabled = enabled;
			onlookerDetectionCheckBox.Enabled = enabled;
			privacyCheckbox.Enabled = enabled && (this.visionCapable || this.testModeCheckbox.Checked);
			monitorAppCheckbox.Enabled = enabled;
		}

		private async Task<bool> UpdateControlsViaGetOptions(FeatureType featureType)
		{
			bool getOptionsSucceeded = true;
			try
			{
				FeatureSetting featureSetting = await _commClient.GetOptions(featureType, globalMode);
				UpdateControlSettings(featureSetting);
			}
			catch (ErrorException ee)
			{
				logger.Error(ee.ErrorObject.Description);
				MessageBox.Show(ee.ErrorObject.Description, "GetOptions Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
				getOptionsSucceeded = false;
			}
			return getOptionsSucceeded;
		}

		internal async Task<bool> RefreshFeaturesViaGetOptions()
		{
			if (_commClient == null)
			{
				logger.Debug("WCF client is null.");
				_commClient = new ClientBackendLink(featureCallback);
			}

			bool success = await UpdateControlsViaGetOptions(FeatureType.LOCK);
			if (success)
			{
				success = await UpdateControlsViaGetOptions(FeatureType.WAKE);
			}
			if (success)
			{
				success = await UpdateControlsViaGetOptions(FeatureType.NOLOCKONPRESENCE);
			}
			if (success)
			{
				success = await UpdateControlsViaGetOptions(FeatureType.ADAPTIVEDIMMING);
			}
			if (success)
			{
				success = await UpdateControlsViaGetOptions(FeatureType.ONLOOKERDETECTION);
			}
			if (success)
			{
				success = await UpdateControlsViaGetOptions(FeatureType.APPLICATION);
			}
			return success;
		}

		public void UpdatePresenceStatusText(string platformStatus, string policyStatus, string sensorStatus)
		{
			PlatformUserPresenceValueLabel.Text = platformStatus;
			PolicyUserPresenceValueLabel.Text = policyStatus;
			SensorUserPresenceValueLabel.Text = sensorStatus;
		}

		private async void UpdatePresenceStatus()
		{
			if (_commClient == null)
			{
				logger.Debug("CST client is null.");
				_commClient = new ClientBackendLink(featureCallback);
			}

			try
			{
				UpdatePresenceStatusText((await _commClient.GetHumanPresenceStatus(HumanPresenceStatusType.PLATFORM)).ToString(), (await _commClient.GetHumanPresenceStatus(HumanPresenceStatusType.POLICY)).ToString(), (await _commClient.GetHumanPresenceStatus(HumanPresenceStatusType.SENSOR)).ToString());
			}
			catch (ErrorException ee)
			{
				logger.Error(ee.ErrorObject.Description);
				UpdatePresenceStatusText(DATA_CONNECTION_ERROR_TEXT, DATA_CONNECTION_ERROR_TEXT, DATA_CONNECTION_ERROR_TEXT);
			}
			catch (Exception ex)
			{
				logger.Error($"Error getting the human presence status: {ex}");
				UpdatePresenceStatusText(DATA_CONNECTION_ERROR_TEXT, DATA_CONNECTION_ERROR_TEXT, DATA_CONNECTION_ERROR_TEXT);
			}
		}

		internal async void privacyCheckBox_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is CheckBox)
			{
				if (((CheckBox)sender).CheckState == CheckState.Checked)
				{
					try
					{
						await _commClient.SetOption(FeatureType.APPLICATION, FeatureProperty.PrivacyEnabled, true, globalMode);
					}
					catch (Exception ex)
					{
						privacyCheckbox.CheckedChanged -= privacyCheckBox_CheckedChanged;
						privacyCheckbox.CheckState = CheckState.Unchecked;
						privacyCheckbox.CheckedChanged += privacyCheckBox_CheckedChanged;
						logger.Error("Error enabling privacy" + ex.Message);
						MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
					}
				}
				else
				{
					try
					{
						await _commClient.SetOption(FeatureType.APPLICATION, FeatureProperty.PrivacyEnabled, false, globalMode);
					}
					catch (Exception ex)
					{
						privacyCheckbox.CheckedChanged -= privacyCheckBox_CheckedChanged;
						privacyCheckbox.CheckState = CheckState.Unchecked;
						privacyCheckbox.CheckedChanged += privacyCheckBox_CheckedChanged;
						logger.Error("Error disabling privacy" + ex.Message);
						MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
					}
				}
			}
		}

		private void adHonorPowerRequestOnRadioButton_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.PowerRequest, true, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void adHonorPowerRequestOffRadioButton_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is RadioButton && ((RadioButton)sender).Checked)
			{
				Task setTask = _commClient.SetOption(FeatureType.ADAPTIVEDIMMING, FeatureProperty.PowerRequest, false, globalMode);
				HandleSetOptionExceptions(setTask);
			}
		}

		private void enabledEventsListBox_SelectedIndexChanged(object sender, EventArgs e)
		{
			if (sender is ListBox)
			{
				ListBox castSender = (ListBox)sender;
				List<string> requestedEvents = GetSubscribedEvents(castSender.SelectedItems);			
				Task setTask = _commClient.SetOption(FeatureType.APPLICATION, FeatureProperty.EnabledEvents, requestedEvents, globalMode);
				HandleSetOptionExceptions(setTask);

				//reset subscribable displayed data to unintialized values
				if (!requestedEvents.Contains(EventTypeMap.EventSubscriptionNames[EventType.EVENT_CST_EXTENDED_CAPABILITY_CHANGED]))
				{
					this.activeSensorTypeValLabel.Text = DATA_UNAVAILABLE_TEXT;
					this.correlationStatusValueLabel.Text = DATA_UNAVAILABLE_TEXT;
					this.visionCapableValLabel.Text = DATA_UNAVAILABLE_TEXT;
					this.mispredictionCountValLabel.Text = DATA_UNAVAILABLE_TEXT;
					this.odSensorStatusVallabel.Text = DATA_UNAVAILABLE_TEXT;

					if (!requestedEvents.Contains(EventTypeMap.EventSubscriptionNames[EventType.EVENT_BIOMETRIC_SENSOR_STATUS_CHANGED]))
					{
						this.biometricPresenceSensorStatusValueLabel.Text = DATA_UNAVAILABLE_TEXT;
					}
				}

				if (!requestedEvents.Contains(EventTypeMap.EventSubscriptionNames[EventType.PROXIMITY]))
				{
					this.proximityDistanceValueLabel.Text = DATA_UNAVAILABLE_TEXT;
				}

				if (!requestedEvents.Contains(EventTypeMap.EventSubscriptionNames[EventType.EVENT_HUMAN_PRESENCE_CHANGED]))
				{
					UpdatePresenceStatusText(DATA_UNAVAILABLE_TEXT, DATA_UNAVAILABLE_TEXT, DATA_UNAVAILABLE_TEXT);
				}
			}
		}

		private async void monitorAppCheckbox_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is CheckBox)
			{
				bool newValue = ((CheckBox)sender).CheckState == CheckState.Checked;
				try
				{
					await _commClient.SetOption(FeatureType.APPLICATION, FeatureProperty.MonitorApp, newValue, globalMode);
				}
				catch (Exception ex)
				{
					monitorAppCheckbox.CheckedChanged -= monitorAppCheckbox_CheckedChanged;
					monitorAppCheckbox.CheckState = CheckState.Unchecked;
					monitorAppCheckbox.CheckedChanged += monitorAppCheckbox_CheckedChanged;
					logger.Error("Error " + (newValue ? "enabling" : "disabling") + " Monitor App setting: " + ex.Message);
					MessageBox.Show(ex.Message, "OnError", MessageBoxButtons.OK, MessageBoxIcon.Error);
				}
			}
		}

		private void testModeCheckbox_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is CheckBox)
			{
				// "Test Mode" controls the enable state of all app feature selections.
				UpdateAppFeatureControlsVisibility(((CheckBox)sender).CheckState == CheckState.Checked);
			}
		}

		internal async Task RefreshFeatureCapability(FeatureType featureType, EventType eventType)
		{
			if (_commClient == null)
			{
				logger.Debug("Application client is null.");
				_commClient = new ClientBackendLink(featureCallback);
			}

			try
			{
				FeatureCapability featureCapability = await _commClient.GetFeatureCapability(featureType);

				if (featureCapability == null)
				{
					logger.Debug($"Received null service capability");
				}
				else
				{
					logger.Debug($"Updating feature cues for {featureType}");
					_ = Task.Run(() => SampleAppProgram.walwoaForm.UpdateFeaturesCues(featureType, featureCapability, eventType));
				}
			}
			catch (ErrorException ee)
			{
				logger.Error(ee.ErrorObject.Description);
				MessageBox.Show(ee.ErrorObject.Description, "GetFeatureCapability Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
			}
			return;
		}

		internal async Task GetAllFeatureCapabilities(IList<FeatureType> featuresToExclude = null)
		{
			if (_commClient == null)
			{
				logger.Debug("Application client is null.");
				_commClient = new ClientBackendLink(featureCallback);
			}

			try
			{
				IDictionary<FeatureType, FeatureCapability> featureCapabilities = await _commClient.GetAllFeatureCapabilities();
				if (featureCapabilities == null)
				{
					logger.Debug($"Obtained null response from the service");
				}
				else
				{
					logger.Debug($"Received feature capabilities for {featureCapabilities.Count} features");
					foreach (var capability in featureCapabilities)
					{
						if (!(featuresToExclude?.Contains(capability.Key)).GetValueOrDefault(false))
						{
							logger.Debug($"Updating feature cues for {capability.Key}");
							_ = Task.Run(() => this.UpdateFeaturesCues(capability.Key, capability.Value, EventType.EVENT_CST_CAPABILITY_CHANGED));
							_ = Task.Run(() => this.UpdateFeaturesCues(capability.Key, capability.Value, EventType.EVENT_CST_EXTENDED_CAPABILITY_CHANGED));
						}
					}
				}
			}
			catch (ErrorException ee)
			{
				logger.Error(ee.ErrorObject.Description);
				MessageBox.Show(ee.ErrorObject.Description, "GetFeatureCapability Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
			}
		}

		internal Task UpdateFeaturesCues(FeatureType featureType, FeatureCapability featureCapability, EventType eventType)
		{
			if (eventType == EventType.EVENT_CST_CAPABILITY_CHANGED)
			{
				UpdateFeatureCapableCue(featureType, featureCapability.Capable);
			}
			else if (eventType == EventType.EVENT_CST_EXTENDED_CAPABILITY_CHANGED)
			{
				if (featureCapability.ExtendedStatus != null)
				{
					UpdateExtendedCapabilitiesCue(featureType, featureCapability.ExtendedStatus);
				}
			}

			return Task.CompletedTask;
		}

		private void UpdateFeatureCapableCue(FeatureType featureType, bool featureCapable)
		{
			// When the APPLICATION is not capable, the other values are a don't care, so we'll display them as not capable
			System.Drawing.Color textColor = (featureCapable && this.appEnabledCheckBox.Checked) ? System.Drawing.Color.Green : System.Drawing.Color.Black;
			switch (featureType)
			{
				case FeatureType.LOCK:
					{
						this.walLabel.ForeColor = textColor;
					}
					break;
				case FeatureType.WAKE:
					{
						this.woaLabel.ForeColor = textColor;
					}
					break;
				case FeatureType.NOLOCKONPRESENCE:
					{
						this.nlopLabel.ForeColor = textColor;
					}
					break;
				case FeatureType.ONLOOKERDETECTION:
					{
						this.onlookerDetectionLabel.ForeColor = textColor;
					}
					break;
				case FeatureType.ADAPTIVEDIMMING:
					{
						this.adLabel.ForeColor = textColor;
					}
					break;
				case FeatureType.APPLICATION:
					{
						if (featureCapable)
						{
							this.appEnabledCheckBox.Checked = true;

							// When the APPLICATION feature becomes capable, the other capabilities stop
							// being a don't care, so we should now refresh all other capabiltiies.
							_ = GetAllFeatureCapabilities(new FeatureType[] { FeatureType.APPLICATION });
						}
						else
						{
							this.walLabel.ForeColor = textColor;
							this.woaLabel.ForeColor = textColor;
							this.nlopLabel.ForeColor = textColor;
							this.onlookerDetectionLabel.ForeColor = textColor;
							this.adLabel.ForeColor = textColor;
							this.appEnabledCheckBox.Checked = false;
						}
					}
					break;
			}
		}

		private void UpdateExtendedCapabilitiesCue(FeatureType featureType, IDictionary<string,string> extendedStatus)
		{
			switch (featureType)
			{
				case FeatureType.ONLOOKERDETECTION:
					{
						if (extendedStatus.TryGetValue("SensorStatus", out string sensorStatus))
						{
							this.odSensorStatusVallabel.Text = sensorStatus;
						}

						if (extendedStatus.TryGetValue("SnoozeActive", out string currentSnoozeState))
						{
							if (Convert.ToBoolean(currentSnoozeState) && !this.odCancelSnoozeButton.Enabled)
							{
								this.odCancelSnoozeButton.Enabled = true;
								this.snoozeRequested = true;
							}
						}

						if(!this.CheckedInitialOnlookerState)
						{
							// Only run this code once - once we have checked the initial state, we can just use the events
							this.CheckedInitialOnlookerState = true;
							if (extendedStatus.TryGetValue("OnlookerDetected", out string onlookerDetected))
							{
								if(Convert.ToBoolean(onlookerDetected) && !this.odCancelSnoozeButton.Enabled)
								{
									ShowOnlookerPopup();
								}
							}
						}
					}
					break;
				case FeatureType.ADAPTIVEDIMMING:
					{
						if (extendedStatus.TryGetValue("MispredictionCount", out string mispredictionCount))
						{
							this.mispredictionCountValLabel.Text = mispredictionCount;
						}
					}
					break;
				case FeatureType.APPLICATION:
					{
						if (extendedStatus.TryGetValue("ActiveBiometricPresenceSensor", out string activeSensorType))
						{
							this.activeSensorTypeValLabel.Text = activeSensorType;
						}
						if (extendedStatus.TryGetValue("CorrelationStatus", out string correlationStatus))
						{
							this.correlationStatusValueLabel.Text = bool.Parse(correlationStatus) ? "Positive" : "Negative";
						}
						if(extendedStatus.TryGetValue("ActiveBiometricPresenceSensorStatus", out string activeSensorStatus))
						{
							this.biometricPresenceSensorStatusValueLabel.Text = activeSensorStatus;
						}
						if(extendedStatus.TryGetValue("HelperServiceActive", out string helperServiceActive))
						{
							SetHelperState(bool.Parse(helperServiceActive));
						}
						if(extendedStatus.TryGetValue("VisionCapable", out string visionCapableString))
						{
							this.visionCapable = bool.Parse(visionCapableString);
							this.visionCapableValLabel.Text = this.visionCapable ? "Yes" : "No";
							this.privacyCheckbox.Enabled = this.visionCapable || this.testModeCheckbox.Checked;
						}
					}
					break;
			}
		}

		internal async Task SnoozeOnlookerDetection(int snoozeVal)
		{
			if (_commClient == null)
			{
				logger.Debug("Application client is null.");
				_commClient = new ClientBackendLink(featureCallback);
			}

			try
			{
				this.snoozeRequested = false;
				string commandString = CommandType.set_value.ToString() + " " + RequestType.ONLOOKER_SNOOZE.ToString();
				string result = await _commClient.ExecuteCommand(commandString, snoozeVal);
				this.snoozeRequested = Newtonsoft.Json.JsonConvert.DeserializeObject<bool>(result);
				if (this.snoozeRequested)
				{
					this.odCancelSnoozeButton.Enabled = true;
				}
			}
			catch (ErrorException ee)
			{
				logger.Error(ee.ErrorObject.Description);
				MessageBox.Show(ee.ErrorObject.Description, "SnoozeOnlookerDetection Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
			}
			catch (Exception ee)
			{
				logger.Error(ee.Message);
				MessageBox.Show(ee.Message, "SnoozeOnlookerDetection Error", MessageBoxButtons.OK, MessageBoxIcon.Error);
			}
		}
		public void HandleOnlookerClear(string message)
		{
			new Thread(() =>
			{
				MessageBox.Show($"Onlooker {message}", "Onlooker Presence Changed", MessageBoxButtons.OK);
			}).Start();
		}

		public void HandleSnoozeComplete()
		{
			this.snoozeRequested = false;
			this.odCancelSnoozeButton.Enabled = false;
		}

		private void odCancelSnoozeButton_Click(object sender, EventArgs e)
		{
			if (this.snoozeRequested)
			{
				_ = Task.Run(async () => await _commClient.SnoozeOnlookerDetection(0));
			}
			else
			{
				new Thread(() =>
				{
					MessageBox.Show("Onlooker Detection snooze is not currently active", "Snooze Onlooker Detection", MessageBoxButtons.OK);
				}).Start();
			}
		}

		public void ShowOnlookerPopup()
		{
			new Thread(() =>
			{
				OnlookerDetectedForm onlookerPopup = new OnlookerDetectedForm();
				DialogResult okVal = onlookerPopup.ShowDialog();

				if (okVal == DialogResult.OK)
				{
					int snoozeVal = onlookerPopup.snoozeTime;
					new Thread(() =>
					{
						SampleAppProgram.walwoaForm.Invoke(new MethodInvoker(delegate
						{
							_ = Task.Run(async () => await SampleAppProgram.walwoaForm.SnoozeOnlookerDetection(snoozeVal));
						}));
					}).Start();
				}
				onlookerPopup.Dispose();
			}).Start();
		}

		public void SetHelperState(bool state)
		{
			this.helperActiveCheckBox.Checked = state;
		}

		private void humanPresencePollCheckbox_CheckedChanged(object sender, EventArgs e)
		{
			if (sender is CheckBox pollCheckbox)
			{
				if (pollCheckbox.Checked)
				{
					StartHumanPresenceUpdateTask();
				}
				else
				{
					StopHumanPresenceUpdateTask();
				}
			}
		}
	}
}